package net.groupadd.yena.common.configuration.core;


import com.fasterxml.jackson.dataformat.javaprop.JavaPropsMapper;
import net.groupadd.yena.common.configuration.api.Configuration;
import net.groupadd.yena.common.configuration.api.ConfigurationLoader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.NoSuchElementException;
import java.util.Properties;

/**
 * The type Configuration.
 */
class ConfigurationImpl implements Configuration {

    private static final Logger LOGGER = LoggerFactory.getLogger(ConfigurationImpl.class);

    private final ConfigurationLoader delegate;
    private final JavaPropsMapper mapper;

    /**
     * Instantiates a new Configuration.
     *
     * @param delegate the delegate
     */
    ConfigurationImpl(final ConfigurationLoader delegate) {
        this.delegate = delegate;
        this.mapper = new JavaPropsMapper();
    }

    @Override
    public String get(String key, String defaultValue) {
        final String val = get(key);
        return (val != null) ? val : defaultValue;
    }

    @Override
    public String get(String key) {
        return get(key, String.class);
    }

    @Override
    public <T> T get(String key, Class<T> type) {
        T val = null;
        try {
            val = this.delegate.getProvider().getProperty(key, type);
        } catch (NoSuchElementException e) {
            LOGGER.warn("Not found key : " + key);
        }
        return val;
    }

    @Override
    public <T> T get(String key, T defaultValue) {
        final T val = (T) get(key, defaultValue.getClass());
        return (val != null) ? val : defaultValue;
    }

    @Override
    public <T> T mapToPojo(String prefix, Class<T> clz) throws IOException {
        final Properties properties = new Properties();
        getAllAsProperties().entrySet()
                .stream()
                .forEach(entry -> {
                    final String key = (String) entry.getKey();
                    if(key.startsWith(prefix)){
                        final int start = prefix.length()+1;
                        properties.put(key.substring(start) , entry.getValue());
                    }
                });
        return this.mapper.readPropertiesAs(properties, clz);
    }

    @Override
    public <T> T mapToPojo(Class<T> clz) throws IOException {
        return this.mapper.readPropertiesAs(getAllAsProperties(),clz);
    }


    @Override
    public Properties getAllAsProperties() {
        return this.delegate.getProvider().allConfigurationAsProperties();
    }
}
