package net.groupadd.yena.common.configuration.impl;

import lombok.extern.slf4j.Slf4j;
import net.groupadd.yena.common.configuration.api.Configuration;
import net.groupadd.yena.common.configuration.api.Provider;
import org.cfg4j.provider.ConfigurationProviderBuilder;
import org.cfg4j.source.ConfigurationSource;
import org.cfg4j.source.context.filesprovider.ConfigFilesProvider;
import org.cfg4j.source.files.FilesConfigurationSource;
import org.cfg4j.source.reload.strategy.ImmediateReloadStrategy;
import org.cfg4j.source.reload.strategy.PeriodicalReloadStrategy;

import java.nio.file.Path;
import java.util.NoSuchElementException;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * The type Configuration provider.
 */
@Slf4j
public class ConfigurationProvider implements Provider<Configuration> {

    private Configuration configuration;

    private ConfigurationProvider() {
        load();
    }

    private static class SingletonHolder {
        /**
         * The Provider.
         */
        static final ConfigurationProvider PROVIDER = new ConfigurationProvider();
    }

    /**
     * Gets instance.
     *
     * @return the instance
     */
    public static ConfigurationProvider getInstance() {
        return SingletonHolder.PROVIDER;
    }

    private void load() {
        final ConfigurationLoaderInfoProvider.LoaderInfo loaderInfo = ConfigurationLoaderInfoProvider.getInstance().get();
        final int period = loaderInfo.getPeriod();
        final TimeUnit timeUnit = loaderInfo.getTimeUnit();
        final Set<Path> paths = loaderInfo.getSet();
        final ConfigFilesProvider configFilesProvider = () -> paths;
        final ConfigurationSource source = new FilesConfigurationSource(configFilesProvider);
        final ConfigurationProviderBuilder builder = new ConfigurationProviderBuilder().withConfigurationSource(source);
        final org.cfg4j.provider.ConfigurationProvider provider = (period > 0) ? builder.withReloadStrategy(new PeriodicalReloadStrategy(period, timeUnit)).build() : builder.withReloadStrategy(new ImmediateReloadStrategy()).build();
        this.configuration = new ConfigurationImpl(provider);
    }

    @Override
    public Configuration get() {
        return this.configuration;
    }

    /**
     * The type Configuration.
     */
    class ConfigurationImpl implements Configuration {

        private final org.cfg4j.provider.ConfigurationProvider delegate;

        private ConfigurationImpl(final org.cfg4j.provider.ConfigurationProvider delegate) {
            this.delegate = delegate;
        }

        @Override
        public String get(String key, String defaultValue) {
            final String val = get(key);
            return (val != null) ? val : defaultValue;
        }

        @Override
        public String get(String key) {
            return get(key, String.class);
        }

        @Override
        public <T> T get(String key, Class<T> type) {
            T val = null;
            try {
                val = this.delegate.getProperty(key, type);
            } catch (NoSuchElementException e) {
                log.warn("Not found key : " + key);
            }
            return val;
        }

        @Override
        public <T> T get(String key, T defaultValue) {
            final T val = (T) get(key, defaultValue.getClass());
            return (val != null) ? val : defaultValue;
        }

        @Override
        public <T> T mapToInterface(String prefix, Class<T> clz) {
            return this.delegate.bind(prefix, clz);
        }

        @Override
        public Properties getAllAsProperties() {
            return this.delegate.allConfigurationAsProperties();
        }

    }

}
