package net.groupadd.yena.common.configuration.impl;

import lombok.extern.slf4j.Slf4j;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * The type Abstract configuration path register.
 */
@Slf4j
public abstract class AbstractConfigurationPathRegister {

    private final ConfigurationLoaderInfoProvider.LoaderInfo loaderInfo;
    private static final String DEFAULT_SYSTEM_PROPERTY_KEY = "config";
    private static final String CONFIG_PATH_DELIMITER = ",";

    /**
     * Instantiates a new Abstract configuration path register.
     */
    protected AbstractConfigurationPathRegister() {
        this.loaderInfo = ConfigurationLoaderInfoProvider.getInstance().get();
        addSystemPropertyKey(DEFAULT_SYSTEM_PROPERTY_KEY);
    }

    /**
     * Instantiates a new Abstract configuration path register.
     *
     * @param period the period
     * @param unit   the unit
     */
    protected AbstractConfigurationPathRegister(final int period, final TimeUnit unit) {
        this();
        this.loaderInfo.setPeriod(period);
        this.loaderInfo.setTimeUnit(unit);

    }

    /**
     * Configure.
     */
    protected abstract void configure();

    /**
     * Add path.
     *
     * @param absoluteFilePaths the absolute file paths
     */
    protected void addPath(final String... absoluteFilePaths) {
        for (String path : absoluteFilePaths) {
            addPath(path);
        }
    }

    /**
     * Add path.
     *
     * @param absoluteFilePath the absolute file path
     */
    protected void addPath(final String absoluteFilePath) {
        final Path path = Paths.get(absoluteFilePath);
        this.loaderInfo.addPath(path);
    }

    /**
     * Add system property key.
     *
     * @param key the key
     */
    protected void addSystemPropertyKey(final String key) {
        final Optional<String> optional = Optional.ofNullable(System.getProperty(key));
        if (optional.isPresent()) {
            addPath(optional.get().split(CONFIG_PATH_DELIMITER));
        } else {
            if (!DEFAULT_SYSTEM_PROPERTY_KEY.equals(key)) {
                log.info("Not found system property value of the key : " + key);
            }
        }
    }

}
