package net.foxgenesis.watame;

import java.io.IOException;
import java.net.ConnectException;
import java.nio.file.Path;
import java.util.Objects;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;

import net.foxgenesis.database.AConnectionProvider;
import net.foxgenesis.database.DatabaseManager;
import net.foxgenesis.database.IDatabaseManager;
import net.foxgenesis.database.providers.MySQLConnectionProvider;
import net.foxgenesis.property.PropertyType;
import net.foxgenesis.property.database.LCKConfigurationDatabase;
import net.foxgenesis.util.MethodTimer;
import net.foxgenesis.util.resource.ResourceUtils;
import net.foxgenesis.watame.plugin.Plugin;
import net.foxgenesis.watame.plugin.PluginHandler;
import net.foxgenesis.watame.plugin.SeverePluginException;
import net.foxgenesis.watame.property.ImmutablePluginProperty;
import net.foxgenesis.watame.property.PluginProperty;
import net.foxgenesis.watame.property.PluginPropertyProvider;
import net.foxgenesis.watame.property.impl.PluginPropertyProviderImpl;

import org.apache.commons.configuration2.ImmutableConfiguration;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

import net.dv8tion.jda.api.JDA;
import net.dv8tion.jda.api.JDA.Status;
import net.dv8tion.jda.api.JDABuilder;
import net.dv8tion.jda.api.OnlineStatus;
import net.dv8tion.jda.api.entities.Activity;
import net.dv8tion.jda.api.exceptions.InvalidTokenException;
import net.dv8tion.jda.api.requests.GatewayIntent;
import net.dv8tion.jda.api.requests.RestAction;
import net.dv8tion.jda.api.utils.ChunkingFilter;
import net.dv8tion.jda.api.utils.MemberCachePolicy;
import net.dv8tion.jda.api.utils.cache.CacheFlag;
import net.dv8tion.jda.internal.utils.IOUtil;

/**
 * Class containing WatameBot implementation
 *
 * @author Ashley
 */
public class WatameBot {
	// ------------------------------- STATIC ====================
	/**
	 * General purpose logger
	 */
	public static final Logger logger = LoggerFactory.getLogger(WatameBot.class);

	/**
	 * Singleton instance of class
	 */
	public static final WatameBot INSTANCE;

	/**
	 * Path pointing to the configuration directory
	 */
	public static final Path CONFIG_PATH;

	/**
	 * Settings that were parsed at startup
	 */
	private static final WatameBotSettings settings;

	/**
	 * watame.ini configuration file
	 */
	private static final ImmutableConfiguration config;

	/**
	 * Push notifications helper
	 */
	private static final PushBullet pushbullet;

	static {
		settings = Main.getSettings();
		config = settings.getConfiguration();

		pushbullet = new PushBullet(settings.getPBToken());
		RestAction.setDefaultFailure(
				err -> pushbullet.pushPBMessage("An Error Occurred in Watame", ExceptionUtils.getStackTrace(err)));

		// Initialize our configuration path
		CONFIG_PATH = settings.configurationPath;

		// Initialize the main bot object with token
		INSTANCE = new WatameBot(settings.getToken());
	}

	// ------------------------------- INSTNACE ====================

	/**
	 * Builder to bulid jda
	 */
	private JDABuilder builder;

	/**
	 * the JDA object
	 */
	private JDA discord;

	// ============================================================================

	/**
	 * Database connection handler
	 */
	private final DatabaseManager manager;

	/**
	 * Database connection provider
	 */
	private AConnectionProvider connectionProvider;

	/**
	 * Plugin configuration database
	 */
	private final LCKConfigurationDatabase propertyDatabase;

	/**
	 * Plugin configuration provider
	 */
	private PluginPropertyProvider propertyProvider;

	/**
	 * Property containing a channel to log messages to
	 */
	private PluginProperty loggingChannel;

	// ============================================================================

	/**
	 * Current state of the bot
	 */
	private State state = State.CONSTRUCTING;

	/**
	 * Plugin handler
	 */
	private final PluginHandler<@NotNull Plugin> pluginHandler;

	/**
	 * Instance context
	 */
	private final Context context;

	/**
	 * Create a new instance with a specified login {@code token}.
	 *
	 * @param token - Token used to connect to discord
	 */
	private WatameBot(String token) {
		// Set shutdown thread
		logger.debug("Adding shutdown hook");
		Runtime.getRuntime().addShutdownHook(new Thread(this::shutdown, "WatameBot Shutdown Thread"));

		// Create our database manager
		manager = new DatabaseManager("Database Manager");

		// Create database connection
		try {
			connectionProvider = getConnectionProvider();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}

		// Create our plugin property database
		propertyDatabase = new LCKConfigurationDatabase(connectionProvider.getDatabase(),
				Constants.DATABASE_TABLE_PROPERTIES, Constants.DATABASE_TABLE_PROPERTY_INFO);
		propertyProvider = new PluginPropertyProviderImpl(propertyDatabase, Constants.PLUGIN_PROPERTY_CACHE_TIME);

		// Create discord connection builder
		builder = createJDA(token, null);

		// Set our instance context
		context = new Context(this, builder, null, pushbullet::pushPBMessage);

		// Create our plugin handler
		pluginHandler = new PluginHandler<>(context, getClass().getModule().getLayer(), Plugin.class);
	}

	void start() throws Exception {
		try {
			long start = System.nanoTime();

			// Update our state to constructing
			updateState(State.CONSTRUCTING);
			logger.info("Starting");
			construct();

			// Set our state to pre-init
			updateState(State.PRE_INIT);
			logger.info("Calling pre-initialization");
			preInit();

			// Set our state to init
			updateState(State.INIT);
			logger.info("Calling initialization");
			init();

			// Set our state to post-init
			updateState(State.POST_INIT);
			logger.info("Calling post-initialization");
			postInit();

			long end = System.nanoTime();

			// Set our state to running
			updateState(State.RUNNING);
			logger.info("Startup completed in {} seconds", MethodTimer.formatToSeconds(end - start));
			logger.info("Calling on ready");
			ready();
		} catch (Exception e) {
			pushbullet.pushPBMessage("An Error Occurred in Watame", ExceptionUtils.getStackTrace(e));
			throw e;
		}
	}

	private void construct() throws Exception {
		/*
		 * ====== CONSTRUCTION ======
		 */
		// Construct plugins
		pluginHandler.loadPlugins();

		// Setup the database
		try {
			Plugin integrated = pluginHandler.getPlugin("integrated");
			if (integrated == null)
				throw new SeverePluginException("Failed to find the integrated plugin!");
			manager.register(integrated, propertyDatabase);
		} catch (IOException e) {
			// Some error occurred while setting up database
			ExitCode.DATABASE_SETUP_ERROR.programExit(e);
		} catch (IllegalArgumentException e) {
			// Resource was null
			ExitCode.DATABASE_INVALID_SETUP_FILE.programExit(e);
		}
	}

	/**
	 * NEED_JAVADOC
	 *
	 * @throws Exception
	 */
	private void preInit() throws Exception {
		// Pre-initialize all plugins
		pluginHandler.preInit();

		logger.info("Starting database pool");
		manager.start(connectionProvider);
	}

	/**
	 * NEED_JAVADOC
	 *
	 * @throws Exception
	 */
	private void init() throws Exception {
		// Assert that the moderation log property is set
		Plugin integrated = pluginHandler.getPlugin("integrated");
		if (integrated != null)
			loggingChannel = propertyProvider.upsertProperty(integrated, "modlog", true, PropertyType.NUMBER);

		// Initialize all plugins
		pluginHandler.init();
	}

	/**
	 * NEED_JAVADOC
	 *
	 * @throws Exception
	 */
	private void postInit() throws Exception {
		/*
		 * ====== POST-INITIALIZATION ======
		 */

		// Post-initialize all plugins
		pluginHandler.postInit(this);

		discord = buildJDA();
		context.onJDABuilder(discord);

		// Register commands
		logger.info("Collecting commands...");
		pluginHandler.updateCommands(discord.updateCommands()).queue();

		/*
		 * ====== END POST-INITIALIZATION ======
		 */

		// Wait for discord to be ready
		if (discord.getStatus() != Status.CONNECTED)
			try {
				// Wait for JDA to be ready for use (BLOCKING!).
				logger.info("Waiting for JDA to be ready...");
				discord.awaitReady();
			} catch (InterruptedException e) {}
		logger.info("Connected to discord!");
	}

	private void ready() {
		// Display our game as ready
		logger.debug("Setting presence to ready");
		discord.getPresence().setPresence(OnlineStatus.ONLINE, Activity
				.playing(config.getString("WatameBot.Status.online", "https://github.com/FoxGenesis/Watamebot")));

		// Fire on ready event
		pluginHandler.onReady(this);
	}

	/**
	 * Bot shutdown method.
	 * <p>
	 * This method will be called on program exit.
	 * </p>
	 */
	private void shutdown() {
		// Set our state to shutdown
		updateState(State.SHUTDOWN);

		System.out.println();
		logger.info("Shutting down...");

		IOUtil.silentClose(pluginHandler);

		// Disconnect from discord
		if (discord != null) {
			logger.info("Shutting down JDA...");
			discord.shutdown();
		}

		// Close connection to datebase
		try {
			logger.info("Closing database connection");
			if (manager != null)
				manager.close();
		} catch (Exception e) {
			logger.error("Error while closing database connection!", e);
		}

		// Await all futures to complete
		if (!ForkJoinPool.commonPool().awaitQuiescence(1, TimeUnit.MINUTES))
			logger.warn("Timed out waiting for common pool shutdown. Continuing shutdown...");

		logger.info("Exiting...");
	}

	/**
	 * Create and connect to discord with specified {@code token} via JDA.
	 *
	 * @param token         - Token used to connect to discord
	 * @param eventExecutor - JDA event pool
	 *
	 * @return connected JDA object
	 */
	private JDABuilder createJDA(String token, ExecutorService eventExecutor) {
		Objects.requireNonNull(token, "Login token must not be null");

		// Setup our JDA with wanted values
		logger.debug("Creating JDA");
		JDABuilder builder = JDABuilder
				.create(token, GatewayIntent.GUILD_MEMBERS, GatewayIntent.GUILD_MODERATION,
						GatewayIntent.GUILD_MESSAGES, GatewayIntent.MESSAGE_CONTENT, GatewayIntent.GUILD_VOICE_STATES)
				.disableCache(CacheFlag.ACTIVITY, CacheFlag.EMOJI, CacheFlag.STICKER, CacheFlag.CLIENT_STATUS,
						CacheFlag.ONLINE_STATUS, CacheFlag.SCHEDULED_EVENTS)
				.setChunkingFilter(ChunkingFilter.ALL).setAutoReconnect(true)
				.setActivity(Activity.playing(config.getString("WatameBot.Status.startup", "Initalizing...")))
				.setMemberCachePolicy(MemberCachePolicy.ALL).setStatus(OnlineStatus.DO_NOT_DISTURB)
				.setEnableShutdownHook(false);

		// Set JDA's event pool executor
		if (eventExecutor != null)
			builder.setEventPool(eventExecutor, true);
		return builder;
	}

	private JDA buildJDA() throws Exception {
		JDA discordTmp = null;

		// Attempt to connect to discord. If failed because no Internet, wait 5 seconds
		// and retry.
		int tries = 0;
		int maxTries = 5;
		double delay = 2;

		while (++tries < maxTries) {
			if (tries > 1) {
				delay = Math.pow(delay, tries);
				logger.warn("Retrying in " + delay + " seconds...");
				Thread.sleep((long) delay * 1000);
			}
			try {
				// Attempt to login to discord
				logger.info("Attempting to login to discord");
				discordTmp = builder.build();

				// We connected. Stop loop.
				break;
			} catch (InvalidTokenException e) {
				ExitCode.INVALID_TOKEN.programExit(e.getLocalizedMessage());
			} catch (Exception ex) {
				// Failed to connect. Log error
				logger.error("Failed to connect: " + ex.getLocalizedMessage());
			}
		}

		if (discordTmp == null) {
			ExitCode.JDA_BUILD_FAIL.programExit("Failed to build JDA after " + maxTries + " tries");
			return null;
		}

		return discordTmp;
	}

	private AConnectionProvider getConnectionProvider() throws Exception {
		AConnectionProvider provider = null;
		int tries = 0;
		int maxTries = 5;
		double delay = 2;
		Properties properties = ResourceUtils.getProperties(settings.configurationPath.resolve("database.properties"),
				Constants.DATABASE_SETTINGS_FILE);

		while (tries < maxTries && provider == null) {
			delay = Math.pow(delay, tries);
			if (delay > 1) {
				try {
					logger.info("Retrying in {} seconds...", delay);
					Thread.sleep((long) delay * 1000);
				} catch (InterruptedException e) {}
			}
			try {
				tries++;
				provider = new MySQLConnectionProvider(properties);
			} catch (ConnectException e) {
				logger.error("Failed to connect to database: {}", e.getLocalizedMessage());
			}
		}

		if (provider == null)
			ExitCode.DATABASE_SETUP_ERROR.programExit("Failed to connect to the database after " + maxTries + " tries");
		return provider;
	}

	/**
	 * Check if this instances {@link JDA} is built and connected to Discord.
	 *
	 * @return {@link JDA} instance is built and its current status is
	 *         {@link Status#CONNECTED}.
	 */
	public boolean isConnectedToDiscord() {
		return discord != null && discord.getStatus() == Status.CONNECTED;
	}

	/**
	 * NEED_JAVADOC
	 *
	 * @return Returns the {@link IDatabaseManager} used to register custom
	 *         databases
	 */
	public IDatabaseManager getDatabaseManager() {
		return manager;
	}

	public PluginPropertyProvider getPropertyProvider() {
		return propertyProvider;
	}

	public ImmutablePluginProperty getLoggingChannel() {
		return loggingChannel;
	}

	/**
	 * NEED_JAVADOC
	 *
	 * @return the current instance of {@link JDA}
	 */
	public JDA getJDA() {
		return discord;
	}

	/**
	 * Get the current state of the bot.
	 *
	 * @return Returns the {@link State} of the bot
	 *
	 * @see State
	 */
	public State getState() {
		return state;
	}

	private void updateState(State state) {
		this.state = state;
		logger.trace("STATE = " + state);
		System.setProperty("watame.status", state.name());
	}

	/**
	 * States {@link WatameBot} goes through on startup.
	 *
	 * @author Ashley
	 */
	public enum State {
		/**
		 * NEED_JAVADOC
		 */
		CONSTRUCTING,
		/**
		 * NEED_JAVADOC
		 */
		PRE_INIT,
		/**
		 * NEED_JAVADOC
		 */
		INIT,
		/**
		 * NEED_JAVADOC
		 */
		POST_INIT,
		/**
		 * WatameBot has finished all loading stages and is running
		 */
		RUNNING,
		/**
		 * WatameBot is shutting down
		 */
		SHUTDOWN;

		public final Marker marker;

		State() {
			marker = MarkerFactory.getMarker(name());
		}
	}
}
