package net.foxgenesis.watame.plugin;

import java.io.Closeable;
import java.util.List;
import java.util.Objects;
import java.util.ServiceLoader;
import java.util.ServiceLoader.Provider;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.function.Consumer;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

import net.dv8tion.jda.api.requests.restaction.CommandListUpdateAction;
import net.foxgenesis.util.CompletableFutureUtils;
import net.foxgenesis.util.MethodTimer;
import net.foxgenesis.watame.Context;
import net.foxgenesis.watame.WatameBot;

/**
 * Class used to handle all plugin related tasks.
 * 
 * @author Ashley
 *
 * @param <T> - the plugin class this instance uses
 */
public class PluginHandler<T extends Plugin> implements Closeable {
	/**
	 * logger
	 */
	private static final Logger logger = LoggerFactory.getLogger(PluginHandler.class);

	/**
	 * Map of plugins
	 */
	private final ConcurrentHashMap<String, T> plugins = new ConcurrentHashMap<>();

	/**
	 * Service loader to load plugins
	 */
	@Nonnull
	private final ServiceLoader<T> loader;

	/**
	 * Module layer of the loader
	 */
	@Nonnull
	private final ModuleLayer layer;

	/**
	 * Class of the plugin we are loading
	 */
	@Nonnull
	private final Class<T> pluginClass;

	/**
	 * Thread pool for loading plugins
	 */
	@Nonnull
	private final ExecutorService pluginExecutor;

	/**
	 * Startup asynchronous executor
	 */
	@Nonnull
	private final Context context;

	/**
	 * Construct a new {@link PluginHandler} with the specified {@link ModuleLayer}
	 * and plugin {@link Class}.
	 * 
	 * @param layer       - layer the {@link ServiceLoader} should use
	 * @param pluginClass - the plugin {@link Class} to load
	 */
	public PluginHandler(Context context, ModuleLayer layer, Class<T> pluginClass) {
		this.context = Objects.requireNonNull(context);
		this.layer = Objects.requireNonNull(layer);
		this.pluginClass = Objects.requireNonNull(pluginClass);

		loader = ServiceLoader.load(layer, pluginClass);
		pluginExecutor = context.getAsynchronousExecutor();
	}

	/**
	 * Load all plugins from the service loader
	 */
	@SuppressWarnings("resource")
	public void loadPlugins() {
		logger.info("Loading plugins...");

		long time = System.nanoTime();

		List<Provider<T>> list = loader.stream().toList();

		logger.debug("Found {} plugins", list.size());

		list.forEach(provider -> {
			logger.debug("Loading {}", provider.type());

			T plugin = provider.get();
			plugins.put(plugin.name, plugin);
			context.getEventRegister().register(plugin);

			logger.info("Loaded {}", plugin.getDisplayInfo());
		});

		time = System.nanoTime() - time;
		logger.info("Loaded all plugins in {}ms", MethodTimer.formatToMilli(time));
	}

	/**
	 * Pre-Initialize all plugins.
	 * 
	 * @return Returns a {@link CompletableFuture} that completes when all plugins
	 *         have finished their {@link Plugin#preInit()}
	 */
	@Nonnull
	public CompletableFuture<Void> preInit() {
		logger.debug("Calling plugin pre-initialization async");
		return forEachPlugin(Plugin::preInit, null);
	}

	/**
	 * Initialize all plugins.
	 * 
	 * @return Returns a {@link CompletableFuture} that completes when all plugins
	 *         have finished their {@link Plugin#init(IEventStore)}
	 */
	@Nonnull
	public CompletableFuture<Void> init() {
		logger.debug("Calling plugin initialization async");
		return forEachPlugin(plugin -> plugin.init(context.getEventRegister()), null);
	}

	/**
	 * Post-Initialize all plugins.
	 * 
	 * @param watamebot - reference to {@link WatameBot} that is passed on to the
	 *                  plugin's {@code postInit}
	 * 
	 * @return Returns a {@link CompletableFuture} that completes when all plugins
	 *         have finished their {@link Plugin#postInit(WatameBot)}
	 */
	@Nonnull
	public CompletableFuture<Void> postInit(WatameBot watamebot) {
		logger.debug("Calling plugin post-initialization async");
		return forEachPlugin(plugin -> plugin.postInit(watamebot), null);
	}

	/**
	 * Post-Initialize all plugins.
	 * 
	 * @param watamebot - reference to {@link WatameBot} that is passed on to the
	 *                  plugin's {@code onReady}o
	 * 
	 * @return Returns a {@link CompletableFuture} that completes when all plugins
	 *         have finished their {@link Plugin#onReady(WatameBot)}
	 */
	@Nonnull
	public CompletableFuture<Void> onReady(WatameBot watamebot) {
		logger.debug("Calling plugin on ready async");
		return forEachPlugin(plugin -> plugin.onReady(watamebot), null);
	}

	/**
	 * Fill a {@link CommandListUpdateAction} will all commands specified by the
	 * loaded plugins.
	 * 
	 * @param action - update task to fill
	 * 
	 * @return Returns the action for chaining
	 */
	@Nonnull
	public CommandListUpdateAction updateCommands(CommandListUpdateAction action) {
		plugins.values().stream().filter(p -> p.providesCommands).map(Plugin::getCommands).forEach(action::addCommands);
		return action;
	}

	/**
	 * Iterate over all plugins that match the {@code filter} and perform a task.
	 * Additionally, any plugin that fires a <b>fatal</b>
	 * {@link SeverePluginException} will be unloaded.
	 * 
	 * @param task   - task that is executed for every plugin in the filter
	 * @param filter - filter to select what plugins to use or {@code null} for all
	 *               plugins
	 * 
	 * @return Returns a {@link CompletableFuture} that completes after all plugins
	 *         have finished the {@code task}.
	 */
	@Nonnull
	private CompletableFuture<Void> forEachPlugin(Consumer<? super T> task, @Nullable Predicate<Plugin> filter) {
		if (filter == null)
			filter = p -> true;
		return CompletableFutureUtils.allOf(plugins.values().stream().filter(filter).map(plugin -> CompletableFuture
				.runAsync(() -> task.accept(plugin), pluginExecutor).exceptionallyAsync(error -> {
					pluginError(plugin, error);
					return null;
				}, pluginExecutor)));
	}

	/**
	 * Remove a plugin from the managed plugins, closing its resources in the
	 * process.
	 * 
	 * @param plugin - the plugin to unload
	 */
	@SuppressWarnings("resource")
	private void unloadPlugin(T plugin) {
		logger.debug("Unloading {}", plugin.getClass());
		plugins.remove(plugin.name);
		context.getEventRegister().unregister(plugin);
		try {
			plugin.close();
		} catch (Exception e) {
			pluginError(plugin, new SeverePluginException(e, false));
		}
		if (plugin.needsDatabase) {
			logger.info("Unloading database connections from ", plugin.getDisplayInfo());
			context.getDatabaseManager().unload(plugin);
		}
		logger.warn(plugin.getDisplayInfo() + " unloaded");
	}

	/**
	 * Indicate that a plugin has thrown an error during one of its initialization
	 * methods.
	 * 
	 * @param plugin - plugin in question
	 * @param error  - the error that was thrown
	 * @param marker - method marker
	 */
	private void pluginError(T plugin, Throwable error) {
		MDC.put("watame.status", context.getState().name());
		Throwable temp = error;

		if (error instanceof CompletionException && error.getCause() instanceof SeverePluginException)
			temp = error.getCause();

		if (temp instanceof SeverePluginException) {
			SeverePluginException pluginException = (SeverePluginException) temp;

			Marker m = MarkerFactory.getMarker(pluginException.isFatal() ? "FATAL" : "SEVERE");

			logger.error(m, "Exception in " + plugin.friendlyName, pluginException);

			if (pluginException.isFatal())
				unloadPlugin(plugin);
		} else
			logger.error("Error in " + plugin.friendlyName, error);
	}

	/**
	 * Close all loaded plugins and <b>wait</b> for the termination of the plugin
	 * thread pool.
	 */
	@Override
	public void close() {
		logger.debug("Closing all pugins");
		forEachPlugin(this::unloadPlugin, null);
	}

	/**
	 * Check if a plugin is loaded.
	 * 
	 * @param identifier - plugin identifier
	 * 
	 * @return Returns {@code true} if the plugin is loaded
	 */
	public boolean isPluginPresent(String identifier) {
		return plugins.containsKey(identifier);
	}

	/**
	 * NEED_JAVADOC
	 * 
	 * @param identifier
	 * 
	 * @return
	 */
	public T getPlugin(String identifier) {
		return plugins.get(identifier);
	}

	/**
	 * Get the class used by this instance.
	 * 
	 * @return Returns a {@link Class} that is used by the {@link ServiceLoader} to
	 *         load the plugins
	 */
	@Nonnull
	public Class<T> getPluginClass() {
		return pluginClass;
	}

	/**
	 * Get the module layer used by this instance.
	 * 
	 * @return Returns a {@link ModuleLayer} that is used by the
	 *         {@link ServiceLoader} to load the plugins
	 */
	@Nonnull
	public ModuleLayer getModuleLayer() {
		return layer;
	}

	/**
	 * NEED_JAVADOC
	 * 
	 * @return
	 */
	@Nonnull
	public ExecutorService getAsynchronousExecutor() {
		return pluginExecutor;
	}
}
