package net.eusashead.parquet.http.serializer.xml;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.http.Charset;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.MediaType;
import net.eusashead.parquet.http.conneg.AcceptableContentType;
import net.eusashead.parquet.http.response.ResponseException;
import net.eusashead.parquet.http.serializer.AbstractSerializer;
import net.eusashead.parquet.http.serializer.Body;
import net.eusashead.parquet.http.serializer.Serializer;
import net.eusashead.parquet.util.Option;

import org.jdom2.Element;
import org.jdom2.Text;
import org.jdom2.output.Format;
import org.jdom2.output.XMLOutputter;

public class JdomXmlSerializer extends AbstractSerializer implements Serializer {

	protected static final MediaType[] mediaTypes = {MediaType.APPLICATION_XML};

	protected static final Charset[] charsets = {Charset.UTF8};

	public JdomXmlSerializer() {
		super(Arrays.asList(mediaTypes), Arrays.asList(charsets));
	}

	@Override
	public Body serialize(Entity target, Set<AcceptableContentType> accept)
			throws ResponseException {

		// Resolve content type
		Option<ContentType> contentType = resolveContentType(accept);

		// Make sure params are valid
		validate(target, contentType);

		// Perform the Serialization
		return serializeXml(target);

	}

	private Body serializeXml(Entity target) throws ResponseException {
		final Element element = renderElement("resource", target);
		StringWriter writer = new StringWriter();
		try {
			final XMLOutputter outputter = new XMLOutputter(Format.getCompactFormat());
			outputter.output(element, writer);
		} catch (IOException e) {
			throw new ResponseException(e);
		}
		return new Body(writer.toString(), new ContentType(MediaType.APPLICATION_XML, Charset.UTF8));
	}

	private Element renderElement(String rel, Entity representation) {

		// Create the root element
		final Element resourceElement = new Element(rel);

		// Add properties
		for (Map.Entry<String, Object> entry : representation.getProperties().entrySet()) {
			Element propertyElement = new Element(entry.getKey());
			if (entry.getValue() != null) {
				propertyElement.setContent(new Text(entry.getValue().toString()));
			} 
			resourceElement.addContent(propertyElement);
		}

		// Add sub-resources
		for (Map.Entry<String, Collection<Entity>> entry : representation.getRelations().entrySet()) {
			for (Entity relation : entry.getValue()) {
				Element subResourceElement = renderElement(entry.getKey(), relation);
				resourceElement.addContent(subResourceElement);
			}
		}

		return resourceElement;
	}

}
