package net.eusashead.parquet.http.serializer.xml;

import java.io.ByteArrayInputStream;
import java.util.Arrays;
import java.util.List;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityBuilder;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.impl.BasicEntityFactory;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.serializer.AbstractDeserializer;
import net.eusashead.parquet.http.serializer.DeserializationException;
import net.eusashead.parquet.http.serializer.Deserializer;

import org.jdom2.Document;
import org.jdom2.Element;
import org.jdom2.Namespace;
import org.jdom2.input.SAXBuilder;
import org.vertx.java.core.buffer.Buffer;


public class JdomXmlDeserializer extends AbstractDeserializer implements Deserializer {

	private static final Namespace XSI_NAMESPACE = Namespace.getNamespace("xsi", "http://www.w3.org/2001/XMLSchema-instance");

	private final static ContentType[] acceptable = {ContentType.APPLICATION_XML_UTF8};

	private final EntityFactory factory;
	
	/**
	 * Create with default {@link BasicEntityFactory}
	 */
	public JdomXmlDeserializer() {
		super(Arrays.asList(acceptable));
		this.factory = new BasicEntityFactory();
	}
	
	/**
	 * Create with cutom {@link EntityFactory}
	 * @param factory {@link EntityFactory} to use
	 */
	public JdomXmlDeserializer(EntityFactory factory) {
		super(Arrays.asList(acceptable));
		this.factory = factory;
	}

	@Override
	public Entity deserialize(Buffer body, ContentType contentType) throws DeserializationException {

		// Check inputs
		validate(body, contentType);

		// Perform deserialization
		return deserializeXml(body.getBytes(), contentType);

	}

	private Entity deserializeXml(byte[] body, ContentType contentType) throws DeserializationException {
		try {
			Document d = new SAXBuilder().build(new ByteArrayInputStream(body));
			Element root = d.getRootElement();
			return readEntity(root);
		} catch (Exception e) {
			throw new DeserializationException(e);
		} 
	}

	private Entity readEntity(Element root) {
		EntityBuilder entity = factory.newEntity();
		readProperties(entity, root);
		readResources(entity, root);
		return entity.build();
	}

	private void readProperties(EntityBuilder entityBuilder, Element element) {
		List<Element> properties = element.getChildren();
		for (Element property : properties) {
			if (property.getChildren().size() == 0) {
				if (property.getAttribute("nil", XSI_NAMESPACE) != null) {
					entityBuilder.property(property.getName(), null);
				} else {
					entityBuilder.property(property.getName(), property.getValue());
				}
			}
		}
	}

	private void readResources(EntityBuilder entityBuilder, Element element) {
		List<Element> resources = element.getChildren();
		for (Element resource : resources) {
			if (resource.getChildren().size() > 0) {
				String rel = resource.getName();
				Entity subResource = readEntity(resource);
				entityBuilder.embed(rel, subResource);
			}
		}
	}

}
