package net.eusashead.parquet.http.serializer.json;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.http.Charset;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.MediaType;
import net.eusashead.parquet.http.conneg.AcceptableContentType;
import net.eusashead.parquet.http.response.ResponseException;
import net.eusashead.parquet.http.serializer.AbstractSerializer;
import net.eusashead.parquet.http.serializer.Body;
import net.eusashead.parquet.http.serializer.Serializer;
import net.eusashead.parquet.util.Option;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.databind.ObjectMapper;

public class JacksonJsonSerializer extends AbstractSerializer implements Serializer {

	protected static final MediaType[] mediaTypes = {MediaType.APPLICATION_JSON};

	protected static final Charset[] charsets = {Charset.UTF8};

	public JacksonJsonSerializer() {
		super(Arrays.asList(mediaTypes), Arrays.asList(charsets));
	}

	@Override
	public Body serialize(Entity target, Set<AcceptableContentType> accept)
			throws ResponseException {

		// Resolve content type
		Option<ContentType> contentType = resolveContentType(accept);

		// Make sure params are valid
		validate(target, contentType);

		// Perform the Serialization
		return serializeJson(target);

	}

	private Body serializeJson(Entity target) throws ResponseException {
		JsonFactory f = new JsonFactory();
		f.setCodec(new ObjectMapper());
		f.enable(JsonGenerator.Feature.QUOTE_FIELD_NAMES);
		StringWriter writer = new StringWriter();
		try {
			JsonGenerator g = f.createGenerator(writer);
			g.writeStartObject();
			renderJson(g, target);
			g.writeEndObject();
			g.close();
		} catch (IOException e) {
			throw new ResponseException(e);
		}

		return new Body(writer.toString(), new ContentType(MediaType.APPLICATION_JSON, Charset.UTF8));
	}

	private void renderJson(JsonGenerator g, Entity representation) throws IOException {

		// Write basic JSON properties
		for (Map.Entry<String, Object> entry : representation.getProperties().entrySet()) {
			if (entry.getValue() != null) {
				g.writeObjectField(entry.getKey(), entry.getValue());
			}
		}

		// Write out embedded Representations
		for (Map.Entry<String, Collection<Entity>> entry : representation.getRelations().entrySet()) {
			if (entry.getValue().size() == 1) {
				if (entry.getValue() != null) {
					for (Entity rep : entry.getValue()) {
						g.writeObjectFieldStart(entry.getKey());
						renderJson(g, rep);
						g.writeEndObject();
					}
				} 
			}
			else {
				g.writeArrayFieldStart(entry.getKey());
				for (Entity subRepresentation : entry.getValue()) {
					g.writeStartObject();
					renderJson(g, subRepresentation);
					g.writeEndObject();
				}
				g.writeEndArray();
			}
		}
	}

}
