package net.eusashead.parquet.http.serializer.json;

import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityBuilder;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.impl.BasicEntityFactory;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.serializer.AbstractDeserializer;
import net.eusashead.parquet.http.serializer.DeserializationException;
import net.eusashead.parquet.http.serializer.Deserializer;

import org.vertx.java.core.buffer.Buffer;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.theoryinpractise.halbuilder.impl.api.Support;

public class JacksonJsonDeserializer extends AbstractDeserializer implements Deserializer {

	private final static ContentType[] acceptable = {ContentType.APPLICATION_JSON_UTF8};

	private final EntityFactory factory;
	
	/**
	 * Create with default {@link BasicEntityFactory}
	 */
	public JacksonJsonDeserializer() {
		super(Arrays.asList(acceptable));
		this.factory = new BasicEntityFactory();
	}
	
	/**
	 * Provide a custom {@link EntityFactory}
	 * @param factory {@link EntityFactory} to use
	 */
	public JacksonJsonDeserializer(EntityFactory factory) {
		super(Arrays.asList(acceptable));
		this.factory = factory;
	}

	@Override
	public Entity deserialize(Buffer body, ContentType contentType) throws DeserializationException {

		// Check inputs
		validate(body, contentType);

		// Perform deserialization
		return deserializeJson(body.getBytes(), contentType);
		
	}

	private Entity deserializeJson(byte[] body, ContentType contentType) throws DeserializationException {
		try {
			ObjectMapper mapper = new ObjectMapper();
			JsonNode rootNode = mapper.readValue(body, JsonNode.class);
			return readEntity(rootNode);
		} catch (Exception e) {
			throw new DeserializationException(e);
		}
	}

	private Entity readEntity(JsonNode rootNode) {
		EntityBuilder entityBuilder = factory.newEntity();
		readProperties(entityBuilder, rootNode);
		readResources(entityBuilder, rootNode);
		return entityBuilder.build();
	}

	private void readProperties(EntityBuilder entityBuilder, JsonNode rootNode) {
		Iterator<String> fieldNames = rootNode.fieldNames();
		while (fieldNames.hasNext()) {
			String fieldName = fieldNames.next();
			if (!Support.RESERVED_JSON_PROPERTIES.contains(fieldName)) {
				JsonNode field = rootNode.get(fieldName);
				if (field.isValueNode()) {
					entityBuilder.property(fieldName, field.isNull() ? null : field.asText());
				}
			}
		}
	}

	private void readResources(EntityBuilder entityBuilder, JsonNode rootNode) {
		Iterator<Map.Entry<String, JsonNode>> fields = rootNode.fields();
		while (fields.hasNext()) {
			Map.Entry<String, JsonNode> keyNode = fields.next();
			if (keyNode.getValue().isArray() || keyNode.getValue().isObject()) {
				if (keyNode.getValue().isArray()) {
					Iterator<JsonNode> values = keyNode.getValue().elements();
					while (values.hasNext()) {
						JsonNode valueNode = values.next();
						entityBuilder.embed(keyNode.getKey(), readEntity(valueNode));
					}
				} else {
					entityBuilder.embed(keyNode.getKey(), readEntity(keyNode.getValue()));
				}
			}
		}
	}
}
