package net.eusashead.parquet.http.serializer.hal;

import java.io.StringReader;
import java.util.Arrays;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.impl.BasicEntityFactory;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.serializer.AbstractDeserializer;
import net.eusashead.parquet.http.serializer.DeserializationException;
import net.eusashead.parquet.http.serializer.Deserializer;

import org.vertx.java.core.buffer.Buffer;

import com.theoryinpractise.halbuilder.api.ReadableRepresentation;
import com.theoryinpractise.halbuilder.api.RepresentationFactory;
import com.theoryinpractise.halbuilder.standard.StandardRepresentationFactory;

public class HalDeserializer extends AbstractDeserializer implements Deserializer {

	private static final ContentType[] acceptable = {
		ContentType.HAL_JSON_UTF8,
		ContentType.HAL_XML_UTF8};
	
	private final RepresentationFactory representationFactory;
	private final EntityFactory entityFactory;
	
	/**
	 * Construct with default {@link StandardRepresentationFactory}
	 * and {@link BasicEntityFactory}
	 */
	public HalDeserializer() {
		super(Arrays.asList(acceptable));
		this.representationFactory = new StandardRepresentationFactory();
		this.entityFactory = new BasicEntityFactory();
	}
	
	/**
	 * Construct with supplied {@link RepresentationFactory}
	 * and default {@link BasicEntityFactory}
	 * @param representationFactory {@link RepresentationFactory} to use
	 */
	public HalDeserializer(RepresentationFactory representationFactory) {
		super(Arrays.asList(acceptable));
		this.representationFactory = representationFactory;
		this.entityFactory = new BasicEntityFactory();
	}
	
	/**
	 * Construct with supplied {@link RepresentationFactory}
	 * and {@link EntityFactory}
	 * @param representationFactory {@link RepresentationFactory} to use
	 * @param entityFactory {@link EntityFactory} to use
	 */
	public HalDeserializer(RepresentationFactory representationFactory, EntityFactory entityFactory) {
		super(Arrays.asList(acceptable));
		this.representationFactory = representationFactory;
		this.entityFactory = entityFactory;
	}
	
	@Override
	public Entity deserialize(Buffer target, ContentType contentType) throws DeserializationException {
		
		// Check arguments
		validate(target, contentType);
		
		// Deserialize!
		try {
			RepresentationEntityConverter converter = new RepresentationEntityConverter();
			ReadableRepresentation representation = representationFactory.readRepresentation(new StringReader(target.toString()));
			return converter.convert(representation, entityFactory).build();
		} catch (IllegalArgumentException e) {
			throw new DeserializationException(e);
		}
	}

}