package net.eusashead.parquet.http.serializer;

import java.util.List;
import java.util.Set;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.conneg.AcceptableContentType;
import net.eusashead.parquet.http.response.ResponseException;
import net.eusashead.parquet.util.Option;

public interface Serializer {
	
	/**
	 * Supported {@link ContentType}
	 * for this {@link Serializer}
	 * @return {@link List} of supported {@link ContentType}
	 */
	List<ContentType> contentTypes();
	
	/**
	 * Does the {@link Serializer}
	 * support the supplied 
	 * {@link ContentType}
	 * @param contentType {@link ContentType} to check
	 * @return true if the {@link ContentType} is supported
	 */
	boolean supports(ContentType contentType);
	
	/**
	 * Serialize the {@link Entity}
	 * into a {@link Body} with the
	 * requested {@link ContentType}
	 * @param target {@link Entity} to serialize
	 * @param contentType {@link ContentType} to serialize into
	 * @return {@link Body} with the required {@link ContentType}
	 * @throws ResponseException 
	 */
	Body serialize(Entity target, Set<AcceptableContentType> contentType) throws ResponseException;

	/**
	 * Resolve the {@link ContentType} 
	 * for the response based on what the
	 * {@link Serializer} can support and
	 * what the client can accept.
	 * 
	 * @return {@link Option} possibly containing a resolved {@link ContentType}
	 */
	Option<ContentType> resolveContentType(Set<AcceptableContentType> accept);

}
