package net.eusashead.parquet.http.serializer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.http.Charset;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.MediaType;
import net.eusashead.parquet.http.conneg.AcceptableContentType;
import net.eusashead.parquet.http.response.ResponseException;
import net.eusashead.parquet.util.Option;

public abstract class AbstractSerializer implements Serializer {
	
	protected final List<ContentType> supported = new ArrayList<>();
	
	public AbstractSerializer() {}
	
	public AbstractSerializer(List<MediaType> mediaTypes, List<Charset> charsets) {
		for (MediaType media : mediaTypes) {
			for (Charset charset : charsets) {
				ContentType contentType = new ContentType(media, charset);
				supported.add(contentType);
			}
		}
	}
	
	@Override
	public Option<ContentType> resolveContentType(Set<AcceptableContentType> accept) {

		for (AcceptableContentType acceptable : accept) {
			for (ContentType supported : this.supported) {
				if (acceptable.contentType().compatible(supported)) {
					return Option.some(supported);
				}
			}
		}
		return Option.none();
	}

	@Override
	public boolean supports(ContentType contentType) {
		for (ContentType candidate : supported) {
			if (contentType.compatible(candidate)) {
				return true;
			}
		}
		return false;
	}
	
	@Override
	public List<ContentType> contentTypes() {
		return Collections.unmodifiableList(supported);
	}
	
	protected void validate(Entity target, Option<ContentType> accept)
			throws ResponseException {
		
		// Cannot serialize a null entity
		if (target == null) {
			throw new ResponseException(HttpStatus.INTERNAL_SERVER_ERROR, "Cannot serialize a null Entity.");
		}
		
		// Is the request content type acceptable?
		if (accept == null || !accept.has()) {
			throw new ResponseException(HttpStatus.NOT_ACCEPTABLE, "Not acceptable");
		}
		
	}

}