package net.eusashead.parquet.http.response.impl;

import io.netty.handler.codec.http.HttpHeaders;

import java.net.URI;
import java.util.Date;
import java.util.Set;
import java.util.TreeSet;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityBuilder;
import net.eusashead.parquet.entity.EntityConverter;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.Link;
import net.eusashead.parquet.entity.PropertyConverter;
import net.eusashead.parquet.hash.HashStrategy;
import net.eusashead.parquet.http.CacheControl;
import net.eusashead.parquet.http.HttpDate;
import net.eusashead.parquet.http.HttpMethod;
import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.etag.ETagValidation;
import net.eusashead.parquet.http.etag.impl.BasicETag;
import net.eusashead.parquet.http.handler.PreconditionSuccessHandler;
import net.eusashead.parquet.http.header.RequestHeaders;
import net.eusashead.parquet.http.request.Request;
import net.eusashead.parquet.http.response.DeleteResponseBuilder;
import net.eusashead.parquet.http.response.ErrorResponseBuilder;
import net.eusashead.parquet.http.response.GetResponseBuilder;
import net.eusashead.parquet.http.response.OptionsResponseBuilder;
import net.eusashead.parquet.http.response.PatchResponseBuilder;
import net.eusashead.parquet.http.response.PostResponseBuilder;
import net.eusashead.parquet.http.response.PutResponseBuilder;
import net.eusashead.parquet.http.response.Response;
import net.eusashead.parquet.http.response.ResponseBuilder;
import net.eusashead.parquet.http.response.ResponseException;
import net.eusashead.parquet.http.serializer.Body;
import net.eusashead.parquet.http.serializer.Serializer;
import net.eusashead.parquet.util.Assert;
import net.eusashead.parquet.util.Option;

import org.vertx.java.core.logging.Logger;

public abstract class ResponseBuilderImpl implements ResponseBuilder, OptionsResponseBuilder, GetResponseBuilder, PutResponseBuilder, PatchResponseBuilder, PostResponseBuilder, DeleteResponseBuilder, ErrorResponseBuilder {

	protected final Request request;
	protected final Response response;
	protected final Serializer serializer;
	protected final EntityFactory entityFactory;
	protected final EntityBuilder entityBuilder;
	protected final Logger logger;
	protected final HashStrategy hashStrategy;

	public ResponseBuilderImpl(Request request, Response response,
			EntityFactory entityFactory, Serializer serializer, 
			HashStrategy hashStrategy, Logger logger) {

		// Check parameters
		validate(request, entityFactory, serializer,
				hashStrategy, logger);

		// Set up fields
		this.request = request;
		this.response = response;
		this.entityFactory = entityFactory;
		this.entityBuilder = entityFactory.newEntity();
		this.serializer = serializer;
		this.hashStrategy = hashStrategy;
		this.logger = logger;

	}

	/**
	 * "Safely" validates parameters
	 * and sends a 500 if invalid
	 * @param request
	 * @param entityFactory
	 * @param serializer
	 * @param contentNegotiator
	 * @param eTagFactory
	 * @param hashStrategy
	 * @param responseStrategy
	 * @param logger
	 */
	private void validate(Request request, EntityFactory entityFactory, 
			Serializer serializer, HashStrategy hashStrategy, Logger logger) {
		try {
			Assert.notNull(request);
			Assert.notNull(serializer);
			Assert.notNull(entityFactory);
			Assert.notNull(hashStrategy);
			Assert.notNull(logger);
		} catch (Exception e) {
			sendError(new ResponseException(e));
		}
	}


	@Override
	public Request request() {
		return this.request;
	}

	@Override
	public ResponseBuilderImpl status(HttpStatus status) {
		this.response.setStatusCode(status.getCode());
		return this;
	}

	@Override
	public ResponseBuilderImpl message(String message) {
		this.response.setStatusMessage(message);
		return this;
	}

	@Override
	public ResponseBuilderImpl eTag() {
		String hash = hashStrategy.encode(this.entityBuilder.build().toString());
		ETag eTag = new BasicETag(hash.getBytes());
		this.response.headers().eTag(eTag);
		return this;
	}

	@Override
	public ResponseBuilderImpl eTag(ETag eTag) {
		this.response.headers().eTag(eTag);
		return this;
	}

	@Override
	public ResponseBuilderImpl lastModified(HttpDate lastModified) {
		this.response.headers().lastModified(lastModified);
		return this;
	}

	@Override
	public ResponseBuilderImpl expires(Long millis) {
		Date now = new Date();
		this.response.headers().add(HttpHeaders.Names.CACHE_CONTROL, new CacheControl(millis).toString());
		this.response.headers().add(HttpHeaders.Names.EXPIRES, new HttpDate(now.getTime() + millis).toString());
		this.response.headers().add(HttpHeaders.Names.DATE, new HttpDate(now.getTime()).toString());
		return this;
	}

	@Override
	public ResponseBuilderImpl location(URI location) {
		this.response.headers().location(location);
		return this;
	}

	@Override
	public OptionsResponseBuilder allow(HttpMethod... methods) {
		this.response.headers().allow(methods);
		return this;
	}	

	@Override
	public ResponseBuilderImpl property(String name, Object value) {
		this.entityBuilder.property(name, value);
		return this;
	}

	@Override
	public ResponseBuilderImpl property(String name, Object value,
			PropertyConverter<?, Object> converter) {
		this.entityBuilder.property(name, value, converter);
		return this;
	}

	@Override
	public ResponseBuilderImpl embed(String rel, Entity entity) {
		this.entityBuilder.embed(rel, entity);
		return this;
	}

	@Override
	public <T> ResponseBuilderImpl embed(String rel, T target,
			EntityConverter<T> converter) {
		this.entityBuilder.embed(rel, target, converter);
		return this;
	}

	@Override
	public <T> ResponseBuilderImpl forEach(String rel, Iterable<T> target,
			EntityConverter<T> converter) {
		this.entityBuilder.forEach(rel, target, converter);
		return this;
	}

	@Override
	public ResponseBuilderImpl link(Link link) {
		this.entityBuilder.link(link);
		return this;
	}

	@Override
	public ResponseBuilderImpl body(Entity entity) {
		this.entityBuilder.append(entity);
		return this;
	}


	@Override
	public <T> ResponseBuilderImpl body(T target, EntityConverter<T> converter) {
		this.entityBuilder.append(target, converter);
		return this;
	}

	@Override
	public EntityFactory entityFactory() {
		return this.entityFactory;
	}

	@Override
	public Response build() {
		return this.response;
	}

	private void sendError(ResponseException e) {

		// Clear the headers
		response.headers().clear();

		// Send the exception
		response.setStatusCode(e.status().getCode()).end();
	}

	protected Body serialize() {

		Body body = null;

		try {

			// Serialize the body
			body = serializer.serialize(entityBuilder.build(), request.acceptableContentType());

			// Add the Content-Type header
			response.headers().contentType(body.contentType());

		} catch (ResponseException e) {
			sendError(e);
		} catch (Exception e) {
			// Log the error
			logger.warn(e.getMessage(), e);
			sendError(new ResponseException(e));
		}

		return body;

	}

	@Override
	public void checkPrecondition(Option<ETag> eTag, 
			Option<HttpDate> lastModified, 
			PreconditionSuccessHandler handler) {
		checkPrecondition(eTag, lastModified, handler, false);
	}

	@Override
	public void requirePrecondition(Option<ETag> eTag, 
			Option<HttpDate> lastModified, 
			PreconditionSuccessHandler handler) {
		checkPrecondition(eTag, lastModified, handler, true);
	}

	public void checkPrecondition(Option<ETag> eTag, 
			Option<HttpDate> lastModified,
			PreconditionSuccessHandler handler, 
			boolean require) {

		final RequestHeaders headers = request.headers();

		// Check precondition exists if required
		if (require && 
				!(headers.hasIfMatch() ||
						headers.hasIfNoneMatch() ||
						headers.hasIfModifiedSince() ||
						headers.hasIfUnmodifiedSince())) {
			request.error(HttpStatus.PRECONDITION_REQUIRED).send();
		} else {

			// Check If-Match
			if (headers.hasIfMatch()) {
				if (!headers.checkIfMatch(eTag, ETagValidation.STRONG)) {
					request.error(HttpStatus.PRECONDITION_FAILED).send();
					return;
				}
			}

			// Check If-Unmodified-Since
			if (headers.hasIfUnmodifiedSince()) {
				if (!headers.checkIfUnmodifiedSince(lastModified)) {
					request.error(HttpStatus.PRECONDITION_FAILED).send();
					return;
				}
			} 

			// Check If-None-Match
			if (headers.hasIfNoneMatch()) {
				if (headers.ifNoneMatch().isWildcard()) {
					if (!headers.hasIfModifiedSince() || 
							!headers.checkIfModifiedSince(lastModified)) {
					request.error(HttpStatus.PRECONDITION_FAILED).send();
					return;
					}
				}
				else { 
					if (!headers.checkIfNoneMatch(eTag, ETagValidation.STRONG)) {
						request.error(HttpStatus.PRECONDITION_FAILED).send();
						return;
					}
				}
			}

			// Check If-Modified-Since
			if (headers.hasIfModifiedSince()) {
				if (!headers.checkIfModifiedSince(lastModified)) {
					request.error(HttpStatus.PRECONDITION_FAILED).send();
					return;
				}
			}

			// Precondition succeeded
			handler.handle();
		}
	}

	/** 
	 * Add the Vary header
	 * @param request
	 */
	protected void addVaryHeader() {

		// Add Vary header for Accept and Accept-Charset
		Set<String> vary = new TreeSet<>();
		vary.add(HttpHeaders.Names.ACCEPT);
		vary.add(HttpHeaders.Names.ACCEPT_CHARSET);

		// Optionally, add Vary for Accept-Encoding and Accept-Language
		if (request.headers().contains(HttpHeaders.Names.ACCEPT_ENCODING)) {
			vary.add(HttpHeaders.Names.ACCEPT_ENCODING);
		}
		if (request.headers().contains(HttpHeaders.Names.ACCEPT_LANGUAGE)) {
			vary.add(HttpHeaders.Names.ACCEPT_LANGUAGE);
		}

		StringBuilder varyBuilder = new StringBuilder();
		int i = 0;
		for (String varyBy : vary) {
			varyBuilder.append(varyBy);
			if (++i < vary.size()) {
				varyBuilder.append(", ");
			}
		}

		request.response().headers().add(HttpHeaders.Names.VARY, varyBuilder.toString());
	}

}