package net.eusashead.parquet.http.response.impl;

import io.netty.handler.codec.http.HttpHeaders;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.hash.HashStrategy;
import net.eusashead.parquet.http.HttpDate;
import net.eusashead.parquet.http.HttpMethod;
import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.etag.ETagValidation;
import net.eusashead.parquet.http.handler.PreconditionSuccessHandler;
import net.eusashead.parquet.http.header.RequestHeaders;
import net.eusashead.parquet.http.request.Request;
import net.eusashead.parquet.http.response.Response;
import net.eusashead.parquet.http.serializer.Body;
import net.eusashead.parquet.http.serializer.Serializer;
import net.eusashead.parquet.util.Option;

import org.vertx.java.core.logging.Logger;

public class GetResponseBuilderImpl extends ResponseBuilderImpl {

	public GetResponseBuilderImpl(Request request, Response response,
			EntityFactory entityFactory, Serializer serializer,
			HashStrategy hashStrategy, Logger logger) {
		super(request, response, entityFactory, serializer,
				hashStrategy, logger);
	}

	@Override
	public void send() {
		
		// Create body
		Body body = serialize();

		// Add vary headers
		addVaryHeader();

		// Check response is not committed first
		if (!response.isCommitted()) {
			// Send body for GET only
			if (request.method().equals(HttpMethod.GET.toString())) {
				response
				.setStatusCode(HttpStatus.OK.getCode())
				.end(body.buffer());
			} else {
				// Add Content-Length for HEAD if not chunked
				if (!response.isChunked()) {
					response.headers().add(HttpHeaders.Names.CONTENT_LENGTH, String.valueOf(body.length()));
				}

				// Send bodiless
				response
				.setStatusCode(HttpStatus.OK.getCode())
				.write("") // TODO hack to stop YokeResponse removing Content-Type header
				.end();
			}
		}

	}

	@Override
	public void checkPrecondition(Option<ETag> eTag, 
			Option<HttpDate> lastModified,
			PreconditionSuccessHandler handler, 
			boolean require) {
		final RequestHeaders headers = request.headers();

		// Check precondition exists if required
		if (require && 
				!(headers.hasIfMatch() ||
						headers.hasIfNoneMatch() ||
						headers.hasIfModifiedSince() ||
						headers.hasIfUnmodifiedSince())) {
			request.error(HttpStatus.PRECONDITION_REQUIRED).send();
		} else {

			// Check If-Match
			if (headers.hasIfMatch()) {
				if (!headers.checkIfMatch(eTag, ETagValidation.WEAK)) {
					request.error(HttpStatus.PRECONDITION_FAILED).send();
					return;
				}
			}

			// Check If-Unmodified-Since
			if (headers.hasIfUnmodifiedSince()) {
				if (!headers.checkIfUnmodifiedSince(lastModified)) {
					request.error(HttpStatus.PRECONDITION_FAILED).send();
					return;
				}
			} 

			// Check If-None-Match
			if (headers.hasIfNoneMatch()) {
				if (headers.ifNoneMatch().isWildcard()) {
					if (!headers.hasIfModifiedSince() || 
							!headers.checkIfModifiedSince(lastModified)) {
						// Add vary headers
						addVaryHeader();
						
						// Send 304
						request
							.response()
							.setStatusCode(HttpStatus.NOT_MODIFIED.getCode())
							.end();
						
						return;
					}
				}
				else { 
					if (!headers.checkIfNoneMatch(eTag, ETagValidation.WEAK)) {
						// Add vary headers
						addVaryHeader();
						
						// Send 304
						request
							.response()
							.setStatusCode(HttpStatus.NOT_MODIFIED.getCode())
							.end();
						
						return;
					}
				}
			}

			// Check If-Modified-Since
			if (headers.hasIfModifiedSince()) {
				if (!headers.checkIfModifiedSince(lastModified)) {
					// Add vary headers
					addVaryHeader();
					
					// Send 304
					request
						.response()
						.setStatusCode(HttpStatus.NOT_MODIFIED.getCode())
						.end();
					
					return;
				}
			}

			// Precondition succeeded
			handler.handle();
		}

	}
}