package net.eusashead.parquet.http.response;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityConverter;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.Link;
import net.eusashead.parquet.entity.MutableEntity;
import net.eusashead.parquet.entity.PropertyConverter;
import net.eusashead.parquet.http.HttpDate;
import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.handler.PreconditionSuccessHandler;
import net.eusashead.parquet.http.request.Request;
import net.eusashead.parquet.util.Option;

public interface ResponseBuilder extends MutableEntity {

	/**
	 * Get the underlying
	 * {@link Request}
	 * @return the current {@link Request}
	 */
	Request request();

	/**
	 * HTTP status code to send
	 * @param status {@link HttpStatus} to use
	 * @return this {@link ResponseBuilder}
	 */
	ResponseBuilder status(HttpStatus status);

	/**
	 * Send the response
	 */
	void send();

	/**
	 * Return the built
	 * {@link Response} 
	 * @return the {@link Response} object
	 */
	Response build();

	/**
	 * Get the {@link EntityFactory}
	 * used to build the body
	 * of this response
	 * @return the {@link EntityFactory} for this {@link ResponseBuilder}
	 */
	EntityFactory entityFactory();

	/**
	 * Set the body to the
	 * supplied {@link Entity}
	 * @param entity
	 * @return this {@link ResponseBuilder}
	 */
	ResponseBuilder body(Entity entity);

	/**
	 * 
	 * @param target
	 * @param converter
	 * @return
	 */
	<T> ResponseBuilder body(T target, EntityConverter<T> converter);

	/**
	 * Check precondition before
	 * executing the {@link PreconditionSuccessHandler}
	 * 
	 * Check value of If-Match, If-Unmodified-Since
	 * If-None-Match and If-Modified-Since and return
	 * either 412 or 304 (for GET or HEAD only) 
	 * if the precondition fails.
	 * 
	 * @param eTag ETag value for current resource
	 * @param lastModified Last-Modified date for current resource 
	 * @param handler {@link PreconditionSuccessHandler} action to execute if precondition succeeds
	 */
	void checkPrecondition(Option<ETag> eTag, 
			Option<HttpDate> lastModified, 
			PreconditionSuccessHandler handler);
	
	/**
	 * Require a precondition check before
	 * executing the {@link PreconditionSuccessHandler}
	 * returning an error if the check fails
	 * or if the precondition headers are absent
	 * 
	 * Check value of If-Match, If-Unmodified-Since
	 * If-None-Match and If-Modified-Since and return
	 * either 412 or 304 (for GET or HEAD only) 
	 * if the precondition fails.
	 * 
	 * If none of the headers above are supplied
	 * return a 428 error.
	 * 
	 * @param eTag ETag value for current resource
	 * @param lastModified Last-Modified date for current resource 
	 * @param handler {@link PreconditionSuccessHandler} action to execute if precondition succeeds
	 */
	void requirePrecondition(Option<ETag> eTag, 
			Option<HttpDate> lastModified, 
			PreconditionSuccessHandler handler);
	
	@Override
	ResponseBuilder property(String name, Object value);

	@Override
	ResponseBuilder property(String name, Object value, PropertyConverter<?,Object> converter);

	@Override
	ResponseBuilder embed(String rel, Entity entity);

	@Override
	<T> ResponseBuilder embed(String rel, T target, EntityConverter<T> converter);

	@Override
	<T> ResponseBuilder forEach(String rel, Iterable<T> target, EntityConverter<T> converter);

	@Override
	ResponseBuilder link(Link link);

}
