package net.eusashead.parquet.http.request.impl;

import io.netty.handler.codec.http.HttpHeaders;

import java.net.InetSocketAddress;
import java.net.URI;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.TreeSet;

import javax.net.ssl.SSLPeerUnverifiedException;
import javax.security.cert.X509Certificate;

import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.hash.HashStrategy;
import net.eusashead.parquet.http.Charset;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.MediaType;
import net.eusashead.parquet.http.conneg.AcceptableCharset;
import net.eusashead.parquet.http.conneg.AcceptableContentType;
import net.eusashead.parquet.http.conneg.AcceptableLanguage;
import net.eusashead.parquet.http.conneg.AcceptableMediaType;
import net.eusashead.parquet.http.conneg.MediaTypeStrategy;
import net.eusashead.parquet.http.header.RequestHeaders;
import net.eusashead.parquet.http.request.Request;
import net.eusashead.parquet.http.response.ErrorResponseBuilder;
import net.eusashead.parquet.http.response.Response;
import net.eusashead.parquet.http.response.ResponseException;
import net.eusashead.parquet.http.response.impl.ErrorResponseBuilderImpl;
import net.eusashead.parquet.http.response.impl.ResponseImpl;
import net.eusashead.parquet.http.serializer.Deserializer;
import net.eusashead.parquet.http.serializer.Serializer;

import org.vertx.java.core.Handler;
import org.vertx.java.core.MultiMap;
import org.vertx.java.core.buffer.Buffer;
import org.vertx.java.core.http.HttpServerFileUpload;
import org.vertx.java.core.http.HttpVersion;
import org.vertx.java.core.logging.Logger;
import org.vertx.java.core.net.NetSocket;

import com.jetdrone.vertx.yoke.middleware.YokeCookie;
import com.jetdrone.vertx.yoke.middleware.YokeRequest;

public abstract class RequestImpl implements Request {

	protected final YokeRequest request;
	protected final RequestHeaders headers;
	protected final Deserializer deserializer;
	protected final Serializer serializer;
	protected final EntityFactory entityFactory;
	protected final MediaTypeStrategy mediaTypeStrategy;
	protected final HashStrategy hashStrategy;
	protected final Response response;
	protected final Logger logger;
	protected final ErrorResponseBuilder errorResponseBuilder;

	public RequestImpl(YokeRequest request, Deserializer deserializer, Serializer serializer, 
			EntityFactory entityFactory, MediaTypeStrategy mediaTypeStrategy,
			HashStrategy hashStrategy, Logger logger) {
		this.request = request;
		this.deserializer = deserializer;
		this.serializer = serializer;
		this.entityFactory = entityFactory;
		this.mediaTypeStrategy = mediaTypeStrategy;
		this.hashStrategy = hashStrategy;
		this.logger = logger;
		this.response = new ResponseImpl(request.response());
		this.headers = new RequestHeaders(request.headers());
		this.errorResponseBuilder = new ErrorResponseBuilderImpl(this, response, entityFactory, serializer, hashStrategy, logger);
	}

	@Override
	public RequestHeaders headers() {
		return this.headers;
	}

	@Override
	public MultiMap params() {
		return request.params();
	}

	@Override
	public Logger logger() {
		return this.logger;
	}

	@Override
	public ErrorResponseBuilder error(HttpStatus status) {
		return this.errorResponseBuilder.status(status);
	}

	/**
	 * Get the acceptable {@link Charset}
	 * for the request in priority
	 * order (q)
	 * @return {@link Set} of {@link AcceptableCharset}
	 * @throws ResponseException
	 */
	@Override
	public Set<AcceptableCharset> acceptableCharsets() {

		// Set of acceptable media types and q values
		Set<AcceptableCharset> acceptable = new TreeSet<>();

		// Header value
		String acceptHeader = headers.get(HttpHeaders.Names.ACCEPT_CHARSET);

		try {
			// Split into all acceptable types
			if (acceptHeader != null) {
				for (String accept : acceptHeader.split(",")){
					AcceptableCharset a = AcceptableCharset.parse(accept);
					acceptable.add(a);
				}
			}
			// Use default encoding
			else {
				AcceptableCharset a = new AcceptableCharset(1.0d, Charset.ANY);
				acceptable.add(a);
			}
		} catch (Exception e) {
			// Ignore any parse exceptions
		}

		return acceptable;
	}

	/**
	 * Get the acceptable {@link Locale}
	 * for the request in priority
	 * order (q)
	 * @return {@link Set} of {@link AcceptableLanguage}
	 * @throws ResponseException
	 */
	@Override
	public Set<AcceptableLanguage> acceptableLanguages() {

		// Set of acceptable media types and q values
		Set<AcceptableLanguage> acceptable = new TreeSet<>();

		// Header value
		String acceptHeader = headers.get(HttpHeaders.Names.ACCEPT_LANGUAGE);

		try {
			// Split into all acceptable types
			if (acceptHeader != null) {
				for (String accept : acceptHeader.split(",")){
					AcceptableLanguage a = AcceptableLanguage.parse(accept);
					acceptable.add(a);
				}
			} else {
				acceptable.add(new AcceptableLanguage(1.0, Locale.getDefault()));
			}
		} catch (Exception e) {
			// Ignore parse exceptions
		}

		return acceptable;
	}

	@Override
	public Set<AcceptableMediaType> acceptableMediaType() {
		try {
			Set<AcceptableMediaType> acceptable = mediaTypeStrategy.acceptableMediaTypes(request);
			if (acceptable.size() == 0) {
				acceptable.add(new AcceptableMediaType(1.0d, MediaType.ANY));
			}
			return acceptable;
		} catch (Exception e) {
			return new TreeSet<>();
		}
	}

	@Override
	public Set<AcceptableContentType> acceptableContentType()
	{
		Set<AcceptableContentType> acceptable = new TreeSet<>();

		for (AcceptableMediaType mediaType : acceptableMediaType()) {
			for (AcceptableCharset charset : acceptableCharsets()) {
				double q = charset.q() * mediaType.q();
				ContentType contentType = new ContentType(mediaType.mediaType(), charset.charset());
				acceptable.add(new AcceptableContentType(q, contentType));
			}
		}
		return acceptable;
	}


	@Override
	public Response response() {
		return this.response;
	}

	@Override
	public HttpVersion version() {
		return this.request.version();
	}

	@Override
	public String method() {
		return this.request.method();
	}

	@Override
	public String uri() {
		return this.request.uri();
	}

	@Override
	public String path() {
		return this.request.path();
	}

	@Override
	public String query() {
		return this.request.query();
	}

	@Override
	public InetSocketAddress remoteAddress() {
		return this.request.remoteAddress();
	}

	@Override
	public InetSocketAddress localAddress() {
		return this.request.localAddress();
	}

	@Override
	public X509Certificate[] peerCertificateChain()
			throws SSLPeerUnverifiedException {
		return this.request.peerCertificateChain();
	}

	@Override
	public URI absoluteURI() {
		return this.request.absoluteURI();
	}

	@Override
	public RequestImpl bodyHandler(Handler<Buffer> bodyHandler) {
		this.request.bodyHandler(bodyHandler);
		return this;
	}

	@Override
	public NetSocket netSocket() {
		return this.request.netSocket();
	}

	@Override
	public RequestImpl expectMultiPart(boolean expect) {
		this.request.expectMultiPart(expect);
		return this;
	}

	@Override
	public RequestImpl uploadHandler(
			Handler<HttpServerFileUpload> uploadHandler) {
		this.request.uploadHandler(uploadHandler);
		return this;
	}

	@Override
	public MultiMap formAttributes() {
		return this.request.formAttributes();
	}

	@Override
	public RequestImpl endHandler(Handler<Void> endHandler) {
		this.request.endHandler(endHandler);
		return this;
	}

	@Override
	public RequestImpl dataHandler(Handler<Buffer> handler) {
		this.request.dataHandler(handler);
		return this;
	}

	@Override
	public RequestImpl pause() {
		this.request.pause();
		return this;
	}

	@Override
	public RequestImpl resume() {
		this.request.resume();
		return this;
	}

	@Override
	public RequestImpl exceptionHandler(Handler<Throwable> handler) {
		this.request.exceptionHandler(handler);
		return this;
	}
	
	@Override
	public <R> R get(String name) {
		return request.get(name);
	}

	@Override
	public <R> R get(String name, R defaultValue) {
		return request.get(name, defaultValue);
	}

	@Override
	public <R> R put(String name, R value) {
		return request.put(name, value);
	}

	@Override
	public String getHeader(String name) {
		return request.getHeader(name);
	}

	@Override
	public List<String> getAllHeaders(String name) {
		return request.getAllHeaders(name);
	}

	@Override
	public String getHeader(String name, String defaultValue) {
		return request.getHeader(name, defaultValue);
	}

	@Override
	public Set<YokeCookie> cookies() {
		return request.cookies();
	}

	@Override
	public YokeCookie getCookie(String name) {
		return request.getCookie(name);
	}

	@Override
	public List<YokeCookie> getAllCookies(String name) {
		return request.getAllCookies(name);
	}

}