package net.eusashead.parquet.http.request;

import java.util.List;
import java.util.Locale;
import java.util.Set;

import net.eusashead.parquet.http.Charset;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.MediaType;
import net.eusashead.parquet.http.conneg.AcceptableCharset;
import net.eusashead.parquet.http.conneg.AcceptableContentType;
import net.eusashead.parquet.http.conneg.AcceptableLanguage;
import net.eusashead.parquet.http.conneg.AcceptableMediaType;
import net.eusashead.parquet.http.conneg.MediaTypeStrategy;
import net.eusashead.parquet.http.header.RequestHeaders;
import net.eusashead.parquet.http.response.ErrorResponseBuilder;
import net.eusashead.parquet.http.response.Response;
import net.eusashead.parquet.http.response.ResponseBuilder;
import net.eusashead.parquet.http.response.ResponseException;

import org.vertx.java.core.http.HttpServerRequest;
import org.vertx.java.core.logging.Logger;

import com.jetdrone.vertx.yoke.middleware.YokeCookie;
import com.jetdrone.vertx.yoke.middleware.YokeRequest;

public interface Request extends HttpServerRequest {
	
	@Override
	RequestHeaders headers();
	
	@Override
	Response response();
	
	/**
	 * Get the {@link ResponseBuilder}
	 * instance for this {@link Request}.
	 * Verb-specific versions will be 
	 * returned by subclasses of this 
	 * interface
	 * @return {@link ResponseBuilder}
	 */
	ResponseBuilder responseBuilder();
	
	/**
	 * Get the Vert.x
	 * {@link Logger}
	 * @return {@link Logger} instance
	 */
	Logger logger();
	
	/**
	 * Get a {@link ErrorResponseBuilder}
	 * for building an error response
	 * @return {@link ErrorResponseBuilder} instance
	 */
	ErrorResponseBuilder error(HttpStatus status);
	
	/**
	 * Get the acceptable Content-Type {@link ContentType}
	 * for the request in priority order (q)  
	 * @return {@link Set} of {@link AcceptableContentType}
	 * @throws ResponseException
	 */
	Set<AcceptableContentType> acceptableContentType();
	
	/**
	 * Get the acceptable media types {@link MediaType}
	 * for the request in priority
	 * order (q) using the {@link MediaTypeStrategy}
	 * registered for this {@link RequestHeaders}
	 * @return {@link Set} of {@link AcceptableMediaType}
	 * @throws ResponseException
	 */
	Set<AcceptableMediaType> acceptableMediaType();

	/**
	 * Get the acceptable languages as
	 * {@link Locale} in priority order (q)
	 * @return {@link Set} of {@link AcceptableLanguage}
	 * @throws ResponseException
	 */
	Set<AcceptableLanguage> acceptableLanguages();

	/**
	 * Get the acceptable {@link Charset} 
	 * in priority order (q)
	 * @return {@link Set} of {@link AcceptableCharset}
	 * @throws ResponseException
	 */
	Set<AcceptableCharset> acceptableCharsets();
	
	/**
	 * @see YokeRequest#get(String)
	 */
	<R> R get(String name);
	
	/**
	 * @see YokeRequest#get(String, Object)
	 */
	<R> R get(String name, R defaultValue);
	
	/**
	 * @see YokeRequest#put(String, Object)
	 */
	<R> R put(String name, R value);
	
	/**
	 * @see YokeRequest#getHeader(String)
	 */
	String getHeader(String name);
	
	/**
	 * @see YokeRequest#getAllHeaders(String)
	 */
	List<String> getAllHeaders(String name);
	
	/**
	 * @see YokeRequest#getHeader(String, String)
	 */
	String getHeader(String name, String defaultValue);
	
	/**
	 * @see YokeRequest#cookies()
	 */
	Set<YokeCookie> cookies();
	
	/**
	 * @see YokeRequest#getCookie(String)
	 */
	YokeCookie getCookie(String name);
	
	/**
	 * @see YokeRequest#getAllCookies(String)
	 */
	List<YokeCookie> getAllCookies(String name);
	
}