package net.eusashead.parquet.http.header;

import io.netty.handler.codec.http.HttpHeaders;

import java.net.URI;
import java.util.Arrays;

import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.HttpDate;
import net.eusashead.parquet.http.HttpMethod;
import net.eusashead.parquet.http.HttpMethods;
import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.etag.impl.BasicETag;
import net.eusashead.parquet.util.Option;

import org.vertx.java.core.MultiMap;

public class ResponseHeaders extends AbstractHeaders {

	public ResponseHeaders(MultiMap headers) {
		super(headers);
	}

	public Option<ETag> eTag() {
		try {
			return Option.some(BasicETag.parse(this.headers.get(HttpHeaders.Names.ETAG)));
		} catch (Exception e) {
			return Option.none();
		}
	}
	
	public ResponseHeaders eTag(ETag eTag) {
		super.headers.add(HttpHeaders.Names.ETAG, eTag.encode());
		return this;
	}

	public Option<HttpDate> lastModified() {
		try {
			HttpDate date = HttpDate.parse(this.headers.get(HttpHeaders.Names.LAST_MODIFIED));
			return Option.some(date);
		} catch (Exception e) {
			return Option.none();
		}
	}

	public ResponseHeaders lastModified(HttpDate date) {
		this.headers.add(HttpHeaders.Names.LAST_MODIFIED, date.toString());
		return this;
	}
	
	public Option<URI> location() {
		try {
			return Option.some(URI.create(this.headers.get(HttpHeaders.Names.LOCATION)));
		} catch (Exception e) {
			return Option.none();
		}
	}
	
	public ResponseHeaders location(URI location) {
		this.headers.add(HttpHeaders.Names.LOCATION, location.toString());
		return this;
	}
	

	public HttpMethods allow() {
		return HttpMethods.parse(this.headers.get(HttpHeaders.Names.ALLOW));
	}
	
	public ResponseHeaders allow(HttpMethod... methods) {
		HttpMethods header = new HttpMethods(Arrays.asList(methods));
		this.headers.add(HttpHeaders.Names.ALLOW, header.toString());
		return this;
	}
	
	public ResponseHeaders allow(HttpMethods methods) {
		this.headers.add(HttpHeaders.Names.ALLOW, methods.toString());
		return this;
	}

	public Option<ContentType> contentType() {
		try {
			return Option.some(ContentType.parse(this.headers.get(HttpHeaders.Names.CONTENT_TYPE)));
		} catch (Exception e) {
			return Option.none();
		}
	}
	
	public ResponseHeaders contentType(ContentType contentType) {
		this.headers.add(HttpHeaders.Names.CONTENT_TYPE, contentType.toString());
		return this;
	}
	
	public boolean hasETag() {
		return has(HttpHeaders.Names.ETAG);
	}
	
	public boolean hasLastModified() {
		return has(HttpHeaders.Names.LAST_MODIFIED);
	}

	public boolean hasContentType() {
		return has(HttpHeaders.Names.CONTENT_TYPE);
	}

	public boolean hasAllow() {
		return has(HttpHeaders.Names.ALLOW);
	}

	public boolean hasLocation() {
		return has(HttpHeaders.Names.LOCATION);
	}

}