package net.eusashead.parquet.http.header;

import io.netty.handler.codec.http.HttpHeaders;
import net.eusashead.parquet.http.ContentType;
import net.eusashead.parquet.http.HttpDate;
import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.etag.ETagValidation;
import net.eusashead.parquet.util.Function;
import net.eusashead.parquet.util.Option;

import org.vertx.java.core.MultiMap;

public class RequestHeaders extends AbstractHeaders {

	public RequestHeaders(MultiMap headers) {
		super(headers);
	}

	public Option<ContentType> contentType() {
		try {
			ContentType contentType = ContentType.parse(headers.get(HttpHeaders.Names.CONTENT_TYPE));
			return Option.some(contentType);
		} catch (Exception e) {
			return Option.none();
		}
	}
	
	public Option<HttpDate> ifModifiedSince() {
		return parseHttpDate(headers.get(HttpHeaders.Names.IF_MODIFIED_SINCE));
	}

	public Option<HttpDate> ifUnmodifiedSince() {
		return parseHttpDate(headers.get(HttpHeaders.Names.IF_UNMODIFIED_SINCE));
	}

	private Option<HttpDate> parseHttpDate(String httpDateString) {
		try {
			HttpDate date = HttpDate.parse(httpDateString);
			return Option.some(date);
		} catch (Exception e) {
			return Option.none();
		}
	}	
	
	/**
	 * Return true if the value of 
	 * the last modification date
	 * is after If-Modified-Since header
	 * @param lastModified
	 * @return true if the last modification date is after the If-Modified-Since header value
	 */
	public boolean checkIfModifiedSince(final Option<HttpDate> lastModified) {
		return ifModifiedSince().maybe(true, new Function<Boolean, HttpDate>() {

			@Override
			public Boolean apply(final HttpDate ifModifiedSince) {
				return lastModified.maybe(true, new Function<Boolean, HttpDate>() {

					@Override
					public Boolean apply(final HttpDate lastModified) {
						return lastModified.after(ifModifiedSince);
					}
				});
			}
		});
	}

	/**
	 * Return true if the value of the
	 * If-Unmodified-Since header is after
	 * the Last-Modified date
	 * @param lastModified
	 * @return
	 */
	public boolean checkIfUnmodifiedSince(final Option<HttpDate> lastModified) {
		
		return ifUnmodifiedSince().maybe(true, new Function<Boolean, HttpDate>() {

			@Override
			public Boolean apply(final HttpDate ifUnmodifiedSince) {
				return lastModified.maybe(false, new Function<Boolean, HttpDate>() {

					@Override
					public Boolean apply(final HttpDate lastModified) {
						return lastModified.equals(ifUnmodifiedSince) || lastModified.before(ifUnmodifiedSince);
					}
				});
			}
		});
	}

	/**
	 * Compare the supplied ETag
	 * with the ETag in the
	 * "If-None-Match" header 
	 * of the incoming request
	 * using the weak equality condition
	 * @param eTag {@link ETag} to check
	 * @param type 
	 * @return
	 */
	public boolean checkIfNoneMatch(Option<ETag> eTag, ETagValidation type) {
		MatchHeader etags = parseEtags(HttpHeaders.Names.IF_NONE_MATCH);
		if (etags.isEmpty()) {
			return true;
		} else {
			return !etags.match(eTag, type);
		}
	}


	/**
	 * Compare the supplied ETag
	 * with the ETag in the
	 * "If-Match" header using
	 * the strong equality condition
	 * of the incoming request
	 * @param eTag
	 * @param type 
	 * @return
	 */
	public boolean checkIfMatch(Option<ETag> eTag, ETagValidation type) {
		MatchHeader etags = parseEtags(HttpHeaders.Names.IF_MATCH);
		if (etags.isEmpty()) {
			return true;
		} else {
			return etags.match(eTag, type);
		}
	}
	
	private MatchHeader parseEtags(final String header) {
		String parse = headers.get(header);
		return MatchHeader.parse(parse);
	}

	public boolean hasIfMatch() {
		return has(HttpHeaders.Names.IF_MATCH);
	}

	public boolean hasIfNoneMatch() {
		return has(HttpHeaders.Names.IF_NONE_MATCH);
	}

	public boolean hasIfModifiedSince() {
		return has(HttpHeaders.Names.IF_MODIFIED_SINCE);
	}

	public boolean hasIfUnmodifiedSince() {
		return has(HttpHeaders.Names.IF_UNMODIFIED_SINCE);
	}

	public RequestHeaders ifModifiedSince(HttpDate httpDate) {
		this.headers.add(HttpHeaders.Names.IF_MODIFIED_SINCE, httpDate.toString());
		return this;
	}

	public RequestHeaders ifUnmodifiedSince(HttpDate httpDate) {
		this.headers.add(HttpHeaders.Names.IF_UNMODIFIED_SINCE, httpDate.toString());
		return this;
	}

	public MatchHeader ifNoneMatch() {
		return this.parseEtags(HttpHeaders.Names.IF_NONE_MATCH);
	}

	public MatchHeader ifMatch() {
		return this.parseEtags(HttpHeaders.Names.IF_MATCH);
	}

	public RequestHeaders ifNoneMatch(ETag eTag) {
		this.headers.add(HttpHeaders.Names.IF_NONE_MATCH, eTag.toString());
		return this;
	}

	public RequestHeaders ifNoneMatch(MatchHeader eTags) {
		this.headers.add(HttpHeaders.Names.IF_NONE_MATCH, eTags.encode());
		return this;
	}

	public RequestHeaders ifMatch(ETag eTag) {
		this.headers.add(HttpHeaders.Names.IF_MATCH, eTag.toString());
		return this;		
	}

	public RequestHeaders ifMatch(MatchHeader eTags) {
		this.headers.add(HttpHeaders.Names.IF_MATCH, eTags.encode());
		return this;
	}

}
