package net.eusashead.parquet.http.header;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.etag.ETagValidation;
import net.eusashead.parquet.http.etag.impl.BasicETag;
import net.eusashead.parquet.util.Function;
import net.eusashead.parquet.util.Option;

import org.apache.commons.lang3.StringUtils;

public class MatchHeader {

	private final String value;

	private MatchHeader(String value) {
		this.value = value;
	}

	public boolean isEmpty() {
		return (this.value == null || this.value.length() == 0);
	}

	public boolean isWildcard() {
		return ("*".equals(this.value));
	}

	public List<ETag> eTags() {
		List<ETag> list = new ArrayList<>();
		List<String> eTags = eTagValues();
		for (String eTag : eTags) {
			if (eTag != null) {
				try {
					list.add(BasicETag.parse(eTag));
				} catch (Exception e) {
					// ignore parse exceptions
				}
			}
		}
		return list;
	}

	private List<String> eTagValues() {
		String[] eTags = new String[0];
		if (value != null) {
			eTags = StringUtils.split(StringUtils.deleteWhitespace(value), ",");
		}
		return Arrays.asList(eTags);
	}

	public boolean matchWeak(final Option<ETag> eTag) {

		if (eTag == null || this.isEmpty()) {
			return false;
		}
		else if (isWildcard()) {
			return true;
		}
		else {
			return eTag.maybe(true, new Function<Boolean, ETag>() {

				@Override
				public Boolean apply(ETag f) {
					for (ETag candidate : eTags()) {
						if (f.weakEquals(candidate)) {
							return Boolean.TRUE;
						}
					}
					return Boolean.FALSE;
				}

			});
		}
	}

	public boolean matchStrong(final Option<ETag> eTag) {

		if (eTag == null || this.isEmpty()) {
			return false;
		}
		else if (isWildcard()) {
			return true;
		}
		else {
			return eTag.maybe(true, new Function<Boolean, ETag>() {

				@Override
				public Boolean apply(ETag f) {
					for (ETag candidate : eTags()) {
						if (f.strongEquals(candidate)) {
							return Boolean.TRUE;
						}
					}
					return Boolean.FALSE;
				}

			});
		} 
	}

	public boolean match(Option<ETag> eTag, ETagValidation type) {
		if (type == ETagValidation.STRONG) {
			return matchStrong(eTag);
		} else {
			return matchWeak(eTag);
		}
	}

	@Override
	public String toString() {
		return encode();
	}

	public String encode() {
		return this.value;
	}

	public static MatchHeader createWildcard() {
		MatchHeader etags = new MatchHeader("*");
		return etags;
	}

	public static MatchHeader createEmpty() {
		MatchHeader etags = new MatchHeader("");
		return etags;
	}

	public static MatchHeader create(List<ETag> list) {
		StringBuilder encode = new StringBuilder();
		for (ETag eTag : list) {
			encode .append(eTag.encode());
			encode.append(", ");
		}
		encode.delete(encode.lastIndexOf(","), encode.length());
		return new MatchHeader(encode.toString());
	}

	public static MatchHeader parse(String header) {
		return new MatchHeader(header);
	}

}
