package net.eusashead.parquet.http.etag.impl;

import java.nio.charset.Charset;
import java.util.Date;

import net.eusashead.parquet.http.etag.ETag;
import net.eusashead.parquet.http.etag.ETagValidation;
import net.eusashead.parquet.util.Assert;

import org.apache.commons.lang3.StringUtils;

public class BasicETag implements ETag {

	private static final String WILDCARD = "*";
	private static final String WEAK_PREFIX = "W/\"";
	private static final String STRONG_PREFIX = "\"";
	private static final String POST_FIX = "\"";

	private final ETagValidation type;
	private final String value;

	public BasicETag(Date date) {
		this(ETagValidation.WEAK, new Long(date.getTime()).toString());
	}

	public BasicETag(Long version) {
		this(ETagValidation.STRONG, version.toString());
	}

	public BasicETag(Integer version) {
		this(ETagValidation.STRONG, version.toString());
	}

	public BasicETag(byte[] hash) {
		this(ETagValidation.STRONG, new String(hash, Charset.forName("utf-8")));
	}

	public BasicETag(ETagValidation type, String value) {
		Assert.notNull(type);
		Assert.notNull(value);
		this.type = type;
		this.value = value;
	}

	/* (non-Javadoc)
	 * @see net.eusashead.vertx.http.ETag#type()
	 */
	@Override
	public ETagValidation type() {
		return this.type;
	}

	/* (non-Javadoc)
	 * @see net.eusashead.vertx.http.ETag#value()
	 */
	@Override
	public String value() {
		return this.value;
	}

	/* (non-Javadoc)
	 * @see net.eusashead.vertx.http.ETag#weakEquals(net.eusashead.vertx.http.ETagImpl)
	 */
	@Override
	public boolean weakEquals(ETag eTag) {
		if (eTag == null) {
			return false;
		}
		if (this.value.equals(WILDCARD) || eTag.value().equals(WILDCARD)) {
			return true;
		}
		return this.value.equals(eTag.value());
	}

	/* (non-Javadoc)
	 * @see net.eusashead.vertx.http.ETag#strongEquals(net.eusashead.vertx.http.ETagImpl)
	 */
	@Override
	public boolean strongEquals(ETag eTag) {
		if (eTag == null) {
			return false;
		}
		if (this.value.equals(WILDCARD) || eTag.value().equals(WILDCARD)) {
			return true;
		}
		if (this.type == ETagValidation.STRONG && eTag.type() == ETagValidation.STRONG) {
			return this.value.equals(eTag.value());
		} else {
			return false;
		}
	}

	/* (non-Javadoc)
	 * @see net.eusashead.vertx.http.ETag#weakEqualsEncoded(java.lang.String)
	 */
	@Override
	public boolean weakEquals(String encoded) {
		return weakEquals(BasicETag.parse(encoded));
	}

	/* (non-Javadoc)
	 * @see net.eusashead.vertx.http.ETag#strongEqualsEncoded(java.lang.String)
	 */
	@Override
	public boolean strongEquals(String encoded) {
		return strongEquals(BasicETag.parse(encoded)); 
	}

	@Override
	public String encode() {
		
		// Create a builder
		StringBuilder builder = new StringBuilder();

		// Is it weak validating?
		if (this.type == ETagValidation.WEAK) {
			builder.append(WEAK_PREFIX);
		} else {
			builder.append(STRONG_PREFIX);
		}

		// Add the value
		builder.append(value);
		builder.append(POST_FIX);

		// Return the string
		return builder.toString();
	}
	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		result = prime * result + ((value == null) ? 0 : value.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		BasicETag other = (BasicETag) obj;
		if (type != other.type)
			return false;
		if (value == null) {
			if (other.value != null)
				return false;
		} else if (!value.equals(other.value))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return encode();
	}


	public static ETag parse(String value) {
		
		BasicETag eTag = null;
		
		if (validate(value)) {
			String parsed;
			ETagValidation validation;
			
			// Is it wildcard, strong or weak
			if (value.equals(WILDCARD)) {
				validation = ETagValidation.STRONG;
				parsed = WILDCARD;		
			}
			else if (value.startsWith(WEAK_PREFIX)) {
				validation = ETagValidation.WEAK;
				parsed = value.substring(WEAK_PREFIX.length(), value.length() - 1);
			} else {
				validation = ETagValidation.STRONG;
				parsed = value.substring(1, value.length() - 1);
			}
			eTag = new BasicETag(validation, parsed);
		} else {
			throw new IllegalArgumentException(String.format("%s is not a valid ETag value.", value));
		}
		
		// Return ETag
		return eTag;
	}
	
	private static boolean validate(String candidate) {
		return (candidate != null) &&
				(candidate.equals(WILDCARD)) ||
				((candidate.length() > 2) &&
				(candidate.endsWith(POST_FIX)) && 
				(candidate.startsWith(WEAK_PREFIX) || candidate.startsWith(STRONG_PREFIX) &&
				(StringUtils.countMatches(candidate, "\"") == 2)));
	}

}