package net.eusashead.parquet.http.container;

import java.util.ArrayList;
import java.util.List;

import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.impl.BasicEntityFactory;
import net.eusashead.parquet.hash.HashStrategy;
import net.eusashead.parquet.hash.impl.Murmur3HashStrategy;
import net.eusashead.parquet.http.HttpMethod;
import net.eusashead.parquet.http.conneg.MediaTypeStrategy;
import net.eusashead.parquet.http.conneg.impl.AcceptHeaderMediaTypeStrategy;
import net.eusashead.parquet.http.handler.DeleteRequestHandler;
import net.eusashead.parquet.http.handler.DeleteRequestHandlerDelegator;
import net.eusashead.parquet.http.handler.GetRequestHandler;
import net.eusashead.parquet.http.handler.GetRequestHandlerDelegator;
import net.eusashead.parquet.http.handler.OptionsRequestHandler;
import net.eusashead.parquet.http.handler.OptionsRequestHandlerDelegator;
import net.eusashead.parquet.http.handler.PatchRequestHandler;
import net.eusashead.parquet.http.handler.PatchRequestHandlerDelegator;
import net.eusashead.parquet.http.handler.PostRequestHandler;
import net.eusashead.parquet.http.handler.PostRequestHandlerDelegator;
import net.eusashead.parquet.http.handler.PutRequestHandler;
import net.eusashead.parquet.http.handler.PutRequestHandlerDelegator;
import net.eusashead.parquet.http.resource.DeleteResource;
import net.eusashead.parquet.http.resource.OptionsResource;
import net.eusashead.parquet.http.resource.PatchResource;
import net.eusashead.parquet.http.resource.PostResource;
import net.eusashead.parquet.http.resource.PutResource;
import net.eusashead.parquet.http.resource.Resource;
import net.eusashead.parquet.http.serializer.Deserializer;
import net.eusashead.parquet.http.serializer.Serializer;
import net.eusashead.parquet.http.serializer.composite.CompositeDeserializer;
import net.eusashead.parquet.http.serializer.composite.CompositeSerializer;
import net.eusashead.parquet.util.Assert;

import org.vertx.java.core.logging.Logger;

import com.jetdrone.vertx.yoke.middleware.Router;

public class RouterBuilder {

	private final Logger logger;
	private final Router router;
	private final CompositeSerializer serializer = new CompositeSerializer();
	private final CompositeDeserializer deserializer = new CompositeDeserializer();

	private MediaTypeStrategy mediaTypeStrategy = new AcceptHeaderMediaTypeStrategy();
	private EntityFactory entityFactory = new BasicEntityFactory();
	private HashStrategy hashStrategy = new Murmur3HashStrategy();
	private boolean traceSupported = false;

	public RouterBuilder(Logger logger) {
		this.logger = logger;

		// Create a RouteMatcher
		this.router = new Router();
	}

	public RouterBuilder registerSerializer(Serializer serializer) {
		Assert.notNull(serializer);
		this.serializer.register(serializer);
		return this;
	}

	public RouterBuilder registerDeserializer(Deserializer deserializer) {
		Assert.notNull(deserializer);
		this.deserializer.register(deserializer);
		return this;
	}

	public RouterBuilder entityFactory(EntityFactory entityFactory) {
		Assert.notNull(entityFactory);
		this.entityFactory = entityFactory;
		return this;
	}

	public RouterBuilder hashStrategy(HashStrategy hashStrategy) {
		Assert.notNull(hashStrategy);
		this.hashStrategy = hashStrategy;
		return this;
	}

	public RouterBuilder mediaTypeStrategy(MediaTypeStrategy mediaTypeStrategy) {
		Assert.notNull(mediaTypeStrategy);
		this.mediaTypeStrategy = mediaTypeStrategy;
		return this;
	}

	public RouterBuilder supportTrace(boolean traceSupported) {
		this.traceSupported = traceSupported;
		return this;
	}

	public Router router() {
		return this.router;
	}

	public RouterBuilder registerResource(Resource resource) {

		// Check resource is valid
		Assert.notNull(resource, "Resource must not be null.");
		Assert.notEmpty(resource.uri(), "Resource URI must not be null.");

		// TRACE and CONNECT are special cases
		if (!traceSupported) {
			router.trace(".*", new NotImplementedHandler());
		}
		router.connect(".*", new NotImplementedHandler());

		// What verbs are supported
		List<HttpMethod> supported = new ArrayList<>();
		List<HttpMethod> unsupported = new ArrayList<>();

		// All resources must support GET and HEAD
		final GetRequestHandler getHandler = resource.get();
		Assert.notNull(getHandler, "GET Handler must not be null.");
		router.get(resource.uri(), new GetRequestHandlerDelegator(getHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
		router.head(resource.uri(), new GetRequestHandlerDelegator(getHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
		supported.add(HttpMethod.GET);
		supported.add(HttpMethod.HEAD);

		// Does the resource accept PUT?
		if (resource instanceof PutResource) {
			PutResource put = PutResource.class.cast(resource);
			final PutRequestHandler putHandler = put.put();
			Assert.notNull(putHandler, "PUT Handler must not be null.");
			router.put(resource.uri(), new PutRequestHandlerDelegator(putHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
			supported.add(HttpMethod.PUT);
		} else {
			unsupported.add(HttpMethod.PUT);
		}

		// Does the resource accept PATCH?
		if (resource instanceof PatchResource) {
			PatchResource patch = PatchResource.class.cast(resource);
			final PatchRequestHandler patchHandler = patch.patch();
			Assert.notNull(patchHandler, "PATCH Handler must not be null.");
			router.patch(resource.uri(), new PatchRequestHandlerDelegator(patchHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
			supported.add(HttpMethod.PATCH);
		} else {
			unsupported.add(HttpMethod.PATCH);
		}

		// Does the resource accept DELETE?
		if (resource instanceof DeleteResource) {
			DeleteResource delete = DeleteResource.class.cast(resource);
			final DeleteRequestHandler deleteHandler = delete.delete();
			Assert.notNull(deleteHandler, "DELETE Handler must not be null.");
			router.delete(resource.uri(), new DeleteRequestHandlerDelegator(deleteHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
			supported.add(HttpMethod.DELETE);
		} else {
			unsupported.add(HttpMethod.DELETE);
		}

		// Does the resource accept POST?
		if (resource instanceof PostResource) {
			PostResource post = PostResource.class.cast(resource);
			final PostRequestHandler postHandler = post.post();
			Assert.notNull(postHandler, "POST Handler must not be null.");
			router.post(resource.uri(), new PostRequestHandlerDelegator(postHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
			supported.add(HttpMethod.POST);
		} else {
			unsupported.add(HttpMethod.POST);
		}

		// Does the resource support OPTIONS?
		if (resource instanceof OptionsResource) {
			// Add custom OPTIONS handler
			OptionsResource options = OptionsResource.class.cast(resource);
			final OptionsRequestHandler optionsHandler = options.options();
			Assert.notNull(optionsHandler, "OPTIONS Handler must not be null.");
			router.options(resource.uri(), new OptionsRequestHandlerDelegator(optionsHandler, deserializer, serializer, entityFactory, mediaTypeStrategy, hashStrategy, logger));
			supported.add(HttpMethod.OPTIONS);
		} else {
			// Use generic OPTIONS handler
			router.options(resource.uri(), new OptionsHandler(supported));
			supported.add(HttpMethod.OPTIONS);
		}

		// Should we support TRACE
		if (traceSupported) {
			router.trace(resource.uri(), new TraceHandler());
			supported.add(HttpMethod.TRACE);
		}

		// Add MethodNotAllowedHandler for PATCH
		if (unsupported.contains(HttpMethod.PATCH)) {
			router.patch(resource.uri(), new MethodNotAllowedHandler(supported));
		}

		// Add MethodNotAllowedHandler for PUT
		if (unsupported.contains(HttpMethod.PUT)) {
			router.put(resource.uri(), new MethodNotAllowedHandler(supported));
		}

		// Add MethodNotAllowedHandler for POST
		if (unsupported.contains(HttpMethod.POST)) {
			router.post(resource.uri(), new MethodNotAllowedHandler(supported));
		}

		// Add MethodNotAllowedHandler for DELETE
		if (unsupported.contains(HttpMethod.DELETE)) {
			router.delete(resource.uri(), new MethodNotAllowedHandler(supported));
		}

		// Return the builder
		return this;
	}

}