package net.eusashead.parquet.http;

import net.eusashead.parquet.util.Assert;

import org.apache.commons.lang3.StringUtils;

public class MediaType implements Compatible<MediaType> {

	private static final String APPLICATION = "application";
	private static final String WILDCARD = "*";

	public static final MediaType ANY = new MediaType(WILDCARD, WILDCARD);
	public static final MediaType APPLICATION_ANY = new MediaType(APPLICATION, WILDCARD);
	public static final MediaType APPLICATION_JSON = new MediaType(APPLICATION, "json");
	public static final MediaType APPLICATION_XML = new MediaType(APPLICATION, "xml");
	public static final MediaType APPLICATION_HAL_JSON = new MediaType(APPLICATION, "hal+json");
	public static final MediaType APPLICATION_HAL_XML = new MediaType(APPLICATION, "hal+xml");

	private final String type;
	private final String subType;

	public MediaType(String type, String subType) {
		Assert.notNull(type);
		Assert.notNull(subType);
		this.type = type;
		this.subType = subType;
	}
	
	public MediaType(String type) {
		Assert.notNull(type);
		this.type = type;
		this.subType = WILDCARD;
	}

	public String type() {
		return this.type;
	}

	public String subType() {
		return this.subType;
	}

	@Override
	public boolean compatible(MediaType other) {
		if ((this.type.equals(other.type) ||
				this.type.equals(WILDCARD)) &&
				(this.subType.equals(other.subType) ||
						this.subType.equals(WILDCARD))) {
			return true;
		} 
		return false;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((subType == null) ? 0 : subType.hashCode());
		result = prime * result + ((type == null) ? 0 : type.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		MediaType other = (MediaType) obj;
		if (subType == null) {
			if (other.subType != null)
				return false;
		} else if (!subType.equals(other.subType))
			return false;
		if (type == null) {
			if (other.type != null)
				return false;
		} else if (!type.equals(other.type))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return type + "/" + subType; 
	}

	public static MediaType parse(String mediaType) {
		// Split type and sub-type
		String[] types = StringUtils.split(mediaType, "/");

		// Create new ContentType
		if (types.length > 1) {
			String type = types[0].trim();
			String subType = types[1].trim();
			return new MediaType(type, subType);
		} 
		
		// Just ain't valid
		throw new IllegalArgumentException(String.format("Cannot parse media type %s", mediaType));
		
	}

}
