package net.eusashead.parquet.http;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

/**
 * Represents a HTTP header date
 * which is in the format 
 * EEE, dd MMM yyyy HH:mm:ss z
 * and must always be in the 
 * GMT timezone
 * 
 * @author patrickvk
 *
 */
public class HttpDate implements Comparable<HttpDate> {

	public static final String HTTP_DATE_FORMAT = "EEE, dd MMM yyyy HH:mm:ss z";
	private final Date date;

	

	public HttpDate(long time) {
		// First, round to nearest 1000
		// because HTTP header dates
		// do not have nanosecond precision
		long offset = (time >= 0) ? 500 : -500;
		long roundedNumber = (time + offset) / 1000 * 1000;
		this.date = new Date(roundedNumber);
	}
	
	public Date date() {
		// Make defensive copy
		return new Date(date.getTime());
	}

	@Override
	public String toString() {
		return getGmtDateFormatter().format(date);
	}
	
	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((date == null) ? 0 : date.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		HttpDate other = (HttpDate) obj;
		if (date == null) {
			if (other.date != null)
				return false;
		} else if (!date.equals(other.date))
			return false;
		return true;
	}

	/**
	 * Parse a HTTP date
	 * from a string using the 
	 * GMT formatter 
	 * @param parse
	 * @return
	 */
	public static HttpDate parse(String parse) throws ParseException {
		return new HttpDate(getGmtDateFormatter().parse(parse).getTime());
	}

	/**
	 * HTTP header dates are always in GMT
	 * @return
	 */
	private static SimpleDateFormat getGmtDateFormatter() {
		final SimpleDateFormat formatter = new SimpleDateFormat(HTTP_DATE_FORMAT);
		final TimeZone GMT = TimeZone.getTimeZone("GMT");
		formatter.setTimeZone(GMT);
		return formatter;
	}

	@Override
	public int compareTo(HttpDate o) {
		return this.date.compareTo(o.date);
	}

	public boolean after(HttpDate date) {
		return this.compareTo(date) == 1;
	}

	public boolean before(HttpDate date) {
		return this.compareTo(date) == -1;
	}

}
