package net.eusashead.parquet.http;

import net.eusashead.parquet.util.Assert;

import org.apache.commons.lang3.StringUtils;

public class ContentType implements Compatible<ContentType> {

	public static final ContentType APPLICATION_JSON_UTF8 = new ContentType(MediaType.APPLICATION_JSON, Charset.UTF8);
	public static final ContentType APPLICATION_XML_UTF8 = new ContentType(MediaType.APPLICATION_XML, Charset.UTF8);
	public static final ContentType HAL_JSON_UTF8 = new ContentType(MediaType.APPLICATION_HAL_JSON, Charset.UTF8);
	public static final ContentType HAL_XML_UTF8 = new ContentType(MediaType.APPLICATION_HAL_XML, Charset.UTF8);
	public static final ContentType APPLICATION_ANY_UTF8 = new ContentType(MediaType.APPLICATION_ANY, Charset.UTF8);
	public static final ContentType ANY_UTF8 = new ContentType(MediaType.ANY, Charset.UTF8);
	public static final ContentType ANY_ANY = new ContentType(MediaType.ANY, Charset.ANY);

	private final MediaType mediaType;
	private final Charset encoding;

	public ContentType(MediaType mediaType, Charset encoding) {
		Assert.notNull(mediaType);
		Assert.notNull(encoding);
		this.mediaType = mediaType; 
		this.encoding = encoding;
	}

	public MediaType mediaType() {
		return this.mediaType;
	}

	public Charset charset() {
		return this.encoding;
	}

	@Override
	public boolean compatible(ContentType other) {
		if ((this.mediaType.compatible(other.mediaType)) &&
				(this.encoding.compatible(other.encoding))) {
			return true;
		} 
		return false;
	}

	/**
	 * Parse a Content-Type header
	 * with the form type/subtype;charset=encoding
	 * ignoring any spaces
	 * @param contentType
	 * @return
	 */
	public static ContentType parse(String contentType) {
		// Strip params and strip whitespace
		String[] parts = StringUtils.split(StringUtils.deleteWhitespace(contentType), ';');
		String media = parts[0].trim();

		// Create MediaType
		MediaType mediaType = MediaType.parse(media);

		// Create Encoding
		Charset charset;
		if (parts.length == 2) {
			charset = Charset.fromContentType(contentType);
		} else {
			charset = Charset.ANY;
		}
		// Create new ContentType
		return new ContentType(mediaType, charset);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((encoding == null) ? 0 : encoding.hashCode());
		result = prime * result
				+ ((mediaType == null) ? 0 : mediaType.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ContentType other = (ContentType) obj;
		if (encoding == null) {
			if (other.encoding != null)
				return false;
		} else if (!encoding.equals(other.encoding))
			return false;
		if (mediaType == null) {
			if (other.mediaType != null)
				return false;
		} else if (!mediaType.equals(other.mediaType))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return mediaType.toString() + "; charset=" + encoding.name();
	}

}
