package net.eusashead.parquet.http;

import org.apache.commons.lang3.StringUtils;

/**
 * Helper for handling Charset encoding
 *  
 * @author patrickvk
 *
 */
public class Charset implements Comparable<Charset>, Compatible<Charset> {

	private static final String WILDCARD = "*";

	public final static Charset ANY = new Charset("*");
	public final static Charset UTF8 = new Charset("utf-8");

	private final String name;

	public Charset(String name) {
		this.name = name;
	}

	public String name() {
		return this.name;
	}

	@Override
	public boolean compatible(Charset other) {
		if (this.name.equals(other.name) ||
				this.name.equals(WILDCARD)) {
			return true;
		} 
		return false;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((name == null) ? 0 : name.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		Charset other = (Charset) obj;
		if (name == null) {
			if (other.name != null)
				return false;
		} else if (!name.equals(other.name))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return this.name;
	}

	public static Charset fromContentType(String contentTypeHeader) {
		String[] parts = StringUtils.split(StringUtils.deleteWhitespace(contentTypeHeader), ';');
		// The charset should be the second part
		if (parts.length > 1 && parts[1] != null) {
			int pos = parts[1].lastIndexOf("charset");
			if (pos == 0) {
				String charsetName = parts[1].substring("charset=".length());
				return new Charset(charsetName);
			} 
		}
		throw new IllegalArgumentException(String.format("Cannot parse Content-Type header %s for charset", contentTypeHeader));
	}

	@Override
	public int compareTo(Charset o) {
		return this.name.compareTo(o.name);
	}

}
