package net.eusashead.parquet.test;

import io.netty.handler.codec.http.HttpHeaders;

import java.util.HashMap;
import java.util.Map;

import net.eusashead.parquet.http.HttpStatus;
import net.eusashead.parquet.http.serializer.Body;
import net.eusashead.parquet.http.serializer.Deserializer;

import org.vertx.java.core.http.HttpClient;
import org.vertx.java.core.http.HttpClientRequest;

public class RequestBuilder<T> {

	private final HttpClient client;
	private final Map<String, String> requestHeaders = new HashMap<>();
	private final Map<String, String> responseHeaders = new HashMap<>();
	private String[] expectHeaders = new String[]{};
	private String uri;
	private HttpStatus status;
	private String requestBody;
	private boolean expectBody;
	private T assertBody;
	private Deserializer deserializer;
	private boolean printBody = false;

	public RequestBuilder(HttpClient client) {
		this.client = client;
	}

	public RequestBuilder<T> uri(String uri) {
		this.uri = uri;
		return this;
	}

	public RequestBuilder<T> assertStatus(HttpStatus status) {
		this.status = status;
		return this;
	}

	public RequestBuilder<T> requestHeader(String header, String value) {
		this.requestHeaders.put(header, value);
		return this;
	}

	public RequestBuilder<T> assertHeader(String header, String value) {
		this.responseHeaders.put(header, value);
		return this;
	}

	public RequestBuilder<T> expectHeaders(String...headers) {
		this.expectHeaders = headers;
		return this;
	}

	public RequestBuilder<T> requestBody(Body body) {
		this.requestBody = body.buffer().toString();
		this.requestHeaders.put(HttpHeaders.Names.CONTENT_LENGTH, Integer.valueOf(body.buffer().toString().length()).toString());
		this.requestHeaders.put(HttpHeaders.Names.CONTENT_TYPE, body.contentType().toString());
		return this;
	}

	public RequestBuilder<T> expectBody(Deserializer deserializer) {
		this.expectBody = true;
		this.deserializer = deserializer;
		return this;
	}

	public RequestBuilder<T> assertBody(T body) {
		this.assertBody = body;
		return this;
	}

	public RequestBuilder<T> printBody() {
		this.printBody = true;
		return this;
	}

	public void options() {
		HttpClientRequest request = client.options(uri, handler());
		setRequestHeaders(request);
		request.end();
	}



	public void get() {
		HttpClientRequest request = client.get(uri, handler());
		setRequestHeaders(request);
		request.end();
	}

	public void head() {
		HttpClientRequest request = client.head(uri, handler());
		setRequestHeaders(request);
		request.end();
	}

	public void put() {
		HttpClientRequest request = client.put(uri, handler());
		setRequestHeaders(request);
		request.write(requestBody);
		request.end();
	}

	public void post() {
		HttpClientRequest request = client.post(uri, handler());
		setRequestHeaders(request);
		request.write(requestBody);
		request.end();
	}

	public void delete() {
		HttpClientRequest request = client.delete(uri, handler());
		setRequestHeaders(request);
		request.end();
	}
	
	public void trace() {
		HttpClientRequest request = client.trace(uri, handler());
		setRequestHeaders(request);
		request.end();
	}
	
	public void connect() {
		HttpClientRequest request = client.connect(uri, handler());
		setRequestHeaders(request);
		request.end();
	}

	private ResponseHandler handler() {
		return expectBody ? bodyHandler() : bodilessHandler();
	}

	private BodyResponseHandler bodyHandler() {
		return new BodyResponseHandler(status, expectHeaders, responseHeaders, deserializer, assertBody, printBody);
	}

	private BodilessResponseHandler bodilessHandler() {
		return new BodilessResponseHandler(status, expectHeaders, responseHeaders);
	}

	private void setRequestHeaders(HttpClientRequest request) {
		for (String header : requestHeaders.keySet()) {
			request.headers().add(header, requestHeaders.get(header));
		}
	}

}