package net.eusashead.parquet.entity.impl;

import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityBuilder;
import net.eusashead.parquet.entity.EntityConverter;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.Link;
import net.eusashead.parquet.entity.PropertyConverter;

public class BasicEntityBuilder implements EntityBuilder {

	private final ConcurrentMap<String, Link> links = new ConcurrentHashMap<>();
	private final ConcurrentMap<String, Object> properties = new ConcurrentHashMap<>();
	private final ConcurrentMap<String, Collection<Entity>> relations = new ConcurrentHashMap<>();
	private final EntityFactory factory;

	/**
	 * Create an {@link EntityBuilder}
	 * @param factory {@link EntityFactory} that created this {@link EntityBuilder}
	 */
	BasicEntityBuilder(EntityFactory factory) {
		this.factory = factory;
	}

	/**
	 * Create an {@link EntityBuilder}
	 * with the supplied {@link URI} for
	 * the "self" link
	 * @param self {@link URI} for self link
	 * @param factory {@link EntityFactory} that created this {@link EntityBuilder}
	 */
	BasicEntityBuilder(URI self, EntityFactory factory) {
		if (self != null) {
			links.put(Link.SELF, Link.self(self));
		}
		this.factory = factory;
	}

	/**
	 * Create an {@link EntityBuilder}
	 * and copy the fields from the 
	 * supplied {@link Entity}
	 * @param entity {@link Entity} to clone
	 * @param factory {@link EntityFactory} that created this {@link EntityBuilder}
	 */
	public BasicEntityBuilder(Entity entity, EntityFactory factory) {
		copyEntity(entity);
		this.factory = factory;
	}

	@Override
	public EntityBuilder property(String name, Object value) {
		this.properties.put(name, value);
		return this;
	}

	@Override
	public EntityBuilder property(String name, Object value,
			PropertyConverter<?, Object> converter) {
		this.properties.put(name, converter.convert(value));
		return this;
	}

	@Override
	public EntityBuilder embed(String name, Entity entity) {
		List<Entity> rels = new ArrayList<>();
		rels.add(entity);
		Collection<Entity> existing = relations.putIfAbsent(name, rels);
		if (existing != null) {
			existing.add(entity);
		}
		return this;
	}

	@Override
	public <T> EntityBuilder embed(String name, T target,
			EntityConverter<T> converter) {
		EntityBuilder converted = converter.convert(target, this.factory);
		List<Entity> rels = new ArrayList<>();
		rels.add(converted.build());
		Collection<Entity> existing = relations.putIfAbsent(name, rels);
		if (existing != null) {
			existing.add(converted.build());
		}
		return this;
	}

	@Override
	public EntityBuilder link(Link link) {
		this.links.put(link.rel(), link);
		return this;
	}

	@Override
	public <T> EntityBuilder forEach(String rel, Iterable<T> target,
			EntityConverter<T> converter) {
		List<Entity> rels = new ArrayList<>();
		for (T t : target) {
			EntityBuilder converted = converter.convert(t, factory);
			rels.add(converted.build());
		}
		Collection<Entity> existing = relations.putIfAbsent(rel, rels);
		if (existing != null) {
			existing.addAll(rels);
		}
		return this;
	}

	@Override
	public EntityBuilder append(Entity entity) {
		copyEntity(entity);
		return this;
	}

	@Override
	public <T> EntityBuilder append(T target, EntityConverter<T> converter) {
		copyEntity(converter.convert(target, factory).build());
		return this;
	}
	
	@Override
	public Entity build() {
		return new BasicEntity(this.links, this.properties,
				this.relations, this.factory);
	}

	@Override
	public EntityFactory entityFactory() {
		return this.factory;
	}

	/**
	 * Copy the supplied {@link Entity}
	 * into this {@link EntityBuilder}
	 * @param entity {@link Entity} to copy
	 */
	private void copyEntity(Entity entity) {
		if (entity != null) {
			this.links.putAll(entity.getLinks());
			this.properties.putAll(entity.getProperties());
			this.relations.putAll(entity.getRelations());
		}
	}

	@Override
	public boolean empty() {
		return (links.size() == 0 &&
				properties.size() ==0 &&
				relations.size() ==0);
	}

}