package net.eusashead.parquet.entity.impl;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.text.Format;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;

import net.eusashead.parquet.entity.Entity;
import net.eusashead.parquet.entity.EntityBuilder;
import net.eusashead.parquet.entity.EntityFactory;
import net.eusashead.parquet.entity.Link;
import net.eusashead.parquet.entity.PropertyConverter;

public class BasicEntity implements Entity, Serializable {

	private static final long serialVersionUID = 1570686849513215182L;

	private final Map<String, Link> links;
	private final Map<String, Object> properties;
	private final Map<String, Collection<Entity>> relations;
	private final EntityFactory entityFactory;
	
	BasicEntity(Map<String, Link> links,
			Map<String, Object> properties,
			Map<String, Collection<Entity>> relations,
			EntityFactory entityFactory) {
		this.links = Collections.unmodifiableMap(links);
		this.properties = Collections.unmodifiableMap(properties);
		this.relations = Collections.unmodifiableMap(relations);
		this.entityFactory = entityFactory;
	}

	@Override
	public String getProperty(String name, Format format) {
		return format.format(this.properties.get(name));
	}

	@Override
	public Object getProperty(String name) {
		return this.properties.get(name);
	}
	
	@Override
	public <T> T getProperty(String name, Class<T> convertTo) {
		return convertTo.cast(getProperty(name));
	}

	@Override
	public <T> T getProperty(String name, PropertyConverter<T, Object> converter) {
		return converter.convert(getProperty(name));
	}

	@Override
	public Map<String, Object> getProperties() {
		return this.properties;
	}

	@Override
	public Map<String, Collection<Entity>> getRelations() {
		return relations;
	}

	@Override
	public Collection<Entity> getRelation(String rel) {
		return relations.get(rel);
	}
	
	@Override
	public Map<String, Link> getLinks() {
		return this.links;
	}

	@Override
	public Link getLink(String rel) {
		return this.links.get(rel);
	}	
	
	private void readObject(ObjectInputStream aInputStream) throws ClassNotFoundException, IOException {
		aInputStream.defaultReadObject();
	}

	private void writeObject(ObjectOutputStream aOutputStream) throws IOException {
		aOutputStream.defaultWriteObject();
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((links == null) ? 0 : links.hashCode());
		result = prime * result
				+ ((properties == null) ? 0 : properties.hashCode());
		result = prime * result
				+ ((relations == null) ? 0 : relations.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		BasicEntity other = (BasicEntity) obj;
		if (links == null) {
			if (other.links != null)
				return false;
		} else if (!links.equals(other.links))
			return false;
		if (properties == null) {
			if (other.properties != null)
				return false;
		} else if (!properties.equals(other.properties))
			return false;
		if (relations == null) {
			if (other.relations != null)
				return false;
		} else if (!relations.equals(other.relations))
			return false;
		return true;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("BasicEntity [links=").append(links)
				.append(", properties=").append(properties)
				.append(", relations=").append(relations).append("]");
		return builder.toString();
	}

	@Override
	public EntityBuilder entityBuilder() {
		EntityBuilder builder = entityFactory.newEntity();
		builder.append(this);
		return builder;
	}

}