package net.eusashead.parquet.entity;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

public class UriBuilder {

	private final static String SEPARATOR = "/";

	private final StringBuilder path;

	private final ConcurrentMap<String, List<String>> params;

	private String lastSegment;

	public UriBuilder() {
		path = new StringBuilder();
		params = new ConcurrentHashMap<>();
		lastSegment = "";
	}

	public UriBuilder(String initialPath) {
		path = new StringBuilder(initialPath);
		params = new ConcurrentHashMap<>();
		lastSegment = "";
	}

	public UriBuilder(UriBuilder clone) {
		path = new StringBuilder(clone.path);
		params = new ConcurrentHashMap<>(clone.params);
		lastSegment = clone.lastSegment;
	}

	// Add a trailing slash
	public UriBuilder slash() {

		return slash("");
	}

	public UriBuilder slash(String segment) {

		// Add slash if necessary
		addSlash(segment);

		// Append the segment
		path.append(segment);

		// Remember the last segment
		this.lastSegment = segment;

		// Return the builder
		return this;
	}

	private void addSlash(String segment) {
		if (!(segment.startsWith(SEPARATOR)
				|| lastSegment.endsWith(SEPARATOR))) {
			path.append(SEPARATOR);
		}
	}

	public UriBuilder param(String name, String value) {
		List<String> values = new ArrayList<>();
		values.add(value);
		List<String> existing = params.putIfAbsent(name, values);
		if (existing != null) {
			existing.add(value);
		}
		return this;
	}

	public UriBuilder params(Map<String, List<String>> params) {
		this.params.putAll(params);
		return this;
	}

	public URI build() {

		// Add the query string
		if (params.size() > 0) {
			path.append("?");
			for (Map.Entry<String, List<String>> entry : params.entrySet()) {
				int count = 0;
				for (String val : entry.getValue()) {
					count++;
					path.append(entry.getKey());
					path.append("=");
					path.append(val);
					if (count < params.size()) {
						path.append("&");
					}
				}
			}
		}

		// Build the URI and return
		try {
			return new URI(path.toString());
		} catch (URISyntaxException e) {
			throw new LinkException(e);
		}
	}
}
