package net.eusashead.parquet.entity;

import java.text.Format;
import java.util.Collection;
import java.util.Map;

public interface Entity {
	
	/**
	 * Return all {@link Link}
	 * in a {@link Map} keyed
	 * with the name
	 * @return {@link Map} of {@link Link}
	 */
	Map<String, Link> getLinks();
	
	/**
	 * Get the named {@link Link}
	 * @param rel name of {@link Link}
	 * @return {@link Link} matching name or null
	 */
	Link getLink(String rel);
	
	/** 
	 * Get the named property
	 * as an {@link Object}
	 * @param key name of property
	 * @return property in {@link Object} form
	 */
	Object getProperty(String key);
	
	/**
	 * Get a property with type <T>
	 * by casting to the supplied
	 * class
	 * @param key
	 * @param convertTo
	 * @return
	 */
	<T> T getProperty(String key, Class<T> convertTo);
	
	/**
	 * Get a property with
	 * the type <T> converted
	 * from {@link Object} using
	 * the supplied {@link PropertyConverter}
	 * @param key name of property to convert and return
	 * @param converter {@link PropertyConverter} to perform conversions
	 * @return property converted to type <T>
	 */
	<T> T getProperty(String key, PropertyConverter<T,Object> converter);
	
	/**
	 * Get a property as a formattted
	 * string using the supplied 
	 * {@link Format}
	 * 
	 * @param key
	 * @param format
	 * @return
	 */
	String getProperty(String key, Format format);
	
	/**
	 * Get all properties in a {@link Map}
	 * keyed with the name of the property
	 * @return {@link Map} of properties
	 */
	Map<String, Object> getProperties();

	/**
	 * Get a {@link Map} containing
	 * all related {@link Entity} keyed with
	 * name (rel).
	 * 
	 * @return {@link Map} with {@link Entity}
	 */
	Map<String, Collection<Entity>> getRelations();

	/**
	 * Get the named relation
	 * 
	 * @param rel {@link String} name of the relation to fetch
	 * @return a {@link Collection} of {@link Entity} that have this name
	 */
	Collection<Entity> getRelation(String rel);
	
	/**
	 * Get an {@link EntityBuilder}
	 * pre-filled with the state of
	 * this {@link Entity} to create
	 * a mutable clone as {@link Entity}
	 * instances must be immutable
	 * 
	 * @return {@link EntityBuilder} for this Entity
	 */
	EntityBuilder entityBuilder();
	
}