package net.eusashead.hateoas.hal.response.impl;

/*
 * #[license]
 * spring-halbuilder
 * %%
 * Copyright (C) 2013 Eusa's Head
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * %[license]
 */

import java.util.Date;

import javax.servlet.http.HttpServletRequest;

import net.eusashead.hateoas.hal.adapter.RepresentationWriter;
import net.eusashead.hateoas.hal.response.HalResponseBuilder;
import net.eusashead.hateoas.header.ETagHeaderStrategy;
import net.eusashead.hateoas.response.ResponseBuilder;

import org.springframework.http.ResponseEntity;

import com.theoryinpractise.halbuilder.api.ReadableRepresentation;
import com.theoryinpractise.halbuilder.api.Representation;
import com.theoryinpractise.halbuilder.api.RepresentationFactory;

/**
 * {@link ResponseBuilder} that enables
 * a HAL {@link Representation} to be
 * constructed for the response.
 * 
 * It exposes a similar API to the
 * HALBuilder {@link Representation} 
 * but automates the URI for the "self"
 * link to be that of the incoming request
 * (which is probably what you would want).
 * 
 * It also contains the methods for adding
 * ETag, Last-Modified, Date, Cache-Control
 * and Expires headers to help with caching
 * and bandwidth conservation.
 * 
 * @author patrickvk
 *
 */
public class HalResponseBuilderImpl extends AbstractHalResponseBuilder implements HalResponseBuilder {

	public HalResponseBuilderImpl(RepresentationFactory representationFactory, HttpServletRequest request) {
		super(representationFactory, request);
	}
	
	@Override
	public HalResponseBuilderImpl withProperty(String name, Object value) {
		assertEntity();
		this.entity.withProperty(name, value);
		return this;
	}

	@Override
	public HalResponseBuilderImpl withRepresentation(String rel, ReadableRepresentation resource) {
		assertEntity();
		this.entity.withRepresentation(rel, resource);
		return this;
	}

	@Override
	public HalResponseBuilderImpl withLink(String rel, String href) {
		assertEntity();
		this.entity.withLink(rel, href);
		return this;
	}

	@Override
	public HalResponseBuilderImpl withNamespace(String namespace, String href) {
		assertEntity();
		this.entity.withNamespace(namespace, href);
		return this;
	}

	@Override
	public HalResponseBuilderImpl withBean(Object value) {
		assertEntity();
		this.entity.withBean(value);
		return this;
	}

	@Override
	public HalResponseBuilderImpl withBeanBasedRepresentation(String rel, String href, Object o) {
		assertEntity();
		this.entity.withBeanBasedRepresentation(rel, href, o);
		return this;
	}

	@Override
	public HalResponseBuilderImpl representation(Representation representation) {
		super.setEntity(representation);
		return this;
	}

	@Override
	public HalResponseBuilderImpl convert(Object bean) {
		assertEntity();
		this.entity.withBean(bean);
		return this;
	}

	@Override
	public <T> HalResponseBuilderImpl convert(T bean, RepresentationWriter<T> writer) {
		this.entity = writer.write(bean, representationFactory);
		return this;
	}

	@Override
	public HalResponseBuilderImpl etag() {
		super.setEtagHeader();
		return this;
	}

	@Override
	public HalResponseBuilderImpl etag(ETagHeaderStrategy strategy) {
		super.setEtagHeader(strategy);
		return this;
	}

	@Override
	public HalResponseBuilderImpl etag(Date date) {
		super.setEtagHeader(date);
		return this;
	}

	@Override
	public HalResponseBuilderImpl etag(Integer version) {
		super.setEtagHeader(version);
		return this;
	}
	
	@Override
	public HalResponseBuilderImpl etag(Long version) {
		super.setEtagHeader(version);
		return this;
	}

	@Override
	public HalResponseBuilderImpl lastModified(Date date) {
		super.setLastModifiedHeader(date);
		return this;
	}

	@Override
	public HalResponseBuilderImpl expireIn(long millis) {
		super.setExpiryHeaders(millis);
		return this;
	}

	@Override
	public ResponseEntity<Representation> build() {
		
		// Validate the body
		super.assertEntity();

		// Assemble the response
		return super.buildResponseEntity(this.entity);
	}

}
