package net.eusashead.hateoas.hal.response;

/*
 * #[license]
 * spring-responseentitybuilder
 * %%
 * Copyright (C) 2013 Eusa's Head
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * %[license]
 */

import java.util.Date;

import net.eusashead.hateoas.hal.adapter.RepresentationWriter;
import net.eusashead.hateoas.header.ETagHeaderStrategy;
import net.eusashead.hateoas.response.ResponseBuilder;

import org.springframework.http.ResponseEntity;

import com.theoryinpractise.halbuilder.api.ReadableRepresentation;
import com.theoryinpractise.halbuilder.api.Representation;

/**
 * Create an appropriate 
 * response to a HTTP GET or HEAD request.
 * 
 * The returned {@link ResponseEntity}
 * should have status code 200 and
 * should have a body of type {@link Representation}, 
 * or a null body for a HEAD. It should have
 * ETags, Last-Modified, Expires and 
 * Cache-Control headers.
 *
 * @author patrickvk
 *
 */
public interface HalResponseBuilder extends ResponseBuilder<Representation> {
	
	/**
	 * Add the supplied property
	 * to the wrapped {@link Representation}
	 * @param name
	 * @param value
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder withProperty(String name, Object value);
	
	/**
	 * Add the supplied {@link Representation}
	 * to the wrapped {@link Representation}
	 * as an embedded resource.
	 * @param rel
	 * @param resource
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder withRepresentation(String rel, ReadableRepresentation resource);
	
	/**
	 * Add a link to the 
	 * wrapped {@link Representation}
	 * @param rel
	 * @param href
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder withLink(String rel, String href);
	
	/**
	 * Add a namespace to the
	 * wrapped {@link Representation}
	 * @param namespace
	 * @param href
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder withNamespace(String namespace, String href);
	
	/**
	 * Copy the fields from the supplied
	 * Java Bean as properties of the 
	 * wrapped {@link Representation} (entity)
	 * @param value
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder withBean(Object value);
	
	/**
	 * Add the supplied bean as 
	 * an embedded {@link Representation}
	 * within the wrapped {@link Representation}
	 * with the specified rel and href.
	 * @param rel the name of the embedded representation
	 * @param href the URI of the representation
	 * @param o the Java Bean object to create the representation from
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder withBeanBasedRepresentation(String rel, String href, Object o);
	
	/**
	 * Set the HAL {@link Representation}
	 * that this response represents
	 * @param the {@link Representation} to set
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder representation(Representation representation);
	
	/**
	 * Set the underlying {@link Representation}
	 * by converting the supplied {@link Object}
	 * into a {@link Representation} using the
	 * default conversion mechanism.
	 * 
	 * @param bean the {@link Object} to convert
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder convert(Object bean);

	/**
	 * Set the underlying {@link Representation}
	 * by converting the supplied {@link Object}
	 * into a {@link Representation} using the
	 * supplied {@link RepresentationWriter} 
	 * in favour of the default conversion mechanism.
	 * 
	 * @param bean the {@link Object} to convert
	 * @param writer the {@link RepresentationWriter} to perform the conversion
	 * @return this {@link HalResponseBuilder}
	 */
	<T> HalResponseBuilder convert(T bean, RepresentationWriter<T> writer);
	
	/**
	 * Create ETag header based
	 * on the best available
	 * {@link ETagStrategy}
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder etag();
	
	/**
	 * Create ETag header based
	 * on the supplied
	 * {@link ETagStrategy}
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder etag(ETagHeaderStrategy strategy);

	/**
	 * Create a weak ETag based
	 * on the supplied date
	 * @param date {@link Date} of last modification
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder etag(Date date);

	/**
	 * Create a strong ETag based
	 * on the supplied version
	 * number
	 * @param version {@link Integer} version number
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder etag(Integer version);
	
	/**
	 * Create a strong ETag based
	 * on the supplied version
	 * number
	 * @param version {@link Long} version number
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder etag(Long version);
	
	/**
	 * Set the Last-Modified header
	 * based on the supplied date
	 * @param date {@link Date} of last modification
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder lastModified(Date date);

	
	/**
	 * Set the the Expires and
	 * Cache-Control headers
	 * based on the supplied 
	 * number of milliseconds
	 * @param millis
	 * @return this {@link HalResponseBuilder}
	 */
	HalResponseBuilder expireIn(long millis);
	
}
