package net.eusashead.hateoas.hal.http.converter.module.impl;

/*
 * #[license]
 * spring-halbuilder
 * %%
 * Copyright (C) 2013 Eusa's Head
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * %[license]
 */

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;

import net.eusashead.hateoas.hal.adapter.RepresentationReader;
import net.eusashead.hateoas.hal.adapter.RepresentationWriter;
import net.eusashead.hateoas.hal.http.converter.module.HalHttpMessageConverterModule;

import com.theoryinpractise.halbuilder.api.ReadableRepresentation;
import com.theoryinpractise.halbuilder.api.RepresentationFactory;


/**
 * Implementation of {@link HalHttpMessageConverterModule}
 * that allows manual registration of {@link RepresentationWriter}
 * and {@link RepresentationReader} for a given {@link Class}
 * @author patrickvk
 *
 */
public class ManualAdapterModuleImpl implements HalHttpMessageConverterModule {

	/**
	 * Writers registered with this module
	 */
	private final Map<Class<?>, RepresentationWriter<?>> writers = new HashMap<Class<?>, RepresentationWriter<?>>();

	/**
	 * Readers registered with this module
	 */
	private final Map<Class<?>, RepresentationReader<?>> readers = new HashMap<Class<?>, RepresentationReader<?>>();


	/* (non-Javadoc)
	 * @see net.eusashead.hateoas.converter.hal.module.HalHttpMessageConverterModule#canRead(java.lang.Class)
	 */
	@Override
	public boolean canRead(Class<?> type) {
		return readers.containsKey(type);
	}

	/* (non-Javadoc)
	 * @see net.eusashead.hateoas.converter.hal.module.HalHttpMessageConverterModule#canWrite(java.lang.Class)
	 */
	@Override
	public boolean canWrite(Class<?> type) {
		return writers.containsKey(type);
	}

	/* (non-Javadoc)
	 * @see net.eusashead.hateoas.converter.hal.module.HalHttpMessageConverterModule#write(java.lang.Object, com.theoryinpractise.halbuilder.api.RepresentationFactory)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public ReadableRepresentation write(Object target,
			RepresentationFactory factory) {
		RepresentationWriter<Object> writer = (RepresentationWriter<Object>) writers.get(target.getClass());
		if (writer != null) {
			return writer.write(target, factory);
		}
		throw new IllegalArgumentException(String.format("Cannot write object of type %s.", target.getClass()));
	}

	/* (non-Javadoc)
	 * @see net.eusashead.hateoas.converter.hal.module.HalHttpMessageConverterModule#read(com.theoryinpractise.halbuilder.api.ReadableRepresentation, java.lang.Class)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public Object read(ReadableRepresentation representation, Class<?> type) {
		RepresentationReader<Object> reader = (RepresentationReader<Object>) readers.get(type);
		if (reader != null) {
			return reader.read(representation, type);
		}
		throw new IllegalArgumentException(String.format("Cannot read object of type %s.", type));
	}

	/**
	 * Register a {@link RepresentationReader}
	 * Note: uses a bit of reflection magic that I
	 * feel occasionally a bit nervous about due 
	 * to type erasure but it seems to work.
	 * @param reader
	 */
	public void registerReader(RepresentationReader<?> reader) {

		// Get the type of the writer using reflection
		Class<?> type = getGenericType(reader, RepresentationReader.class);

		// If the type isn't detected, barf
		if (type == null) {
			throw new IllegalArgumentException("Could not determine type of RepresentationWriter<?> using reflection. Please log a bug report.");
		}
		
		// Add the reader to the map
		this.readers.put(type, reader);
	}

	/**
	 * Register a {@link RepresentationWriter}
	 * Note: uses a bit of reflection magic that I
	 * feel occasionally a bit nervous about due 
	 * to type erasure but it seems to work.
	 * @param writer
	 */
	public void registerWriter(RepresentationWriter<?> writer) {

		// Get the type of the writer using reflection
		Class<?> type = getGenericType(writer, RepresentationWriter.class);

		// If the type isn't detected, barf
		if (type == null) {
			throw new IllegalArgumentException("Could not determine type of RepresentationWriter<?> using reflection. Please log a bug report.");
		}

		// Add the writer to the map
		this.writers.put(type, writer);
	}

	private Class<?> getGenericType(Object target, Type targetType) {
		Class<?> type = null;
		Type[] genericInterfaces = target.getClass().getGenericInterfaces();
		for (Type interf : genericInterfaces) {
			ParameterizedType t = (ParameterizedType)interf;
			if (t.getRawType() == targetType) {
				type =  (Class<?>) t.getActualTypeArguments()[0];
			}
		}
		return type;
	}

}
