package net.eusashead.hateoas.hal.http.converter;

/*
 * #[license]
 * spring-halbuilder
 * %%
 * Copyright (C) 2013 Eusa's Head
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * %[license]
 */

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import net.eusashead.hateoas.hal.http.converter.module.HalHttpMessageConverterModule;

import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.MediaType;
import org.springframework.http.converter.AbstractHttpMessageConverter;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;

import com.theoryinpractise.halbuilder.api.ReadableRepresentation;
import com.theoryinpractise.halbuilder.api.Representation;
import com.theoryinpractise.halbuilder.api.RepresentationFactory;

/**
 * {@link HttpMessageConverter} implementation
 * that handles HAL representations 
 * (application/hal+json and (application/hal+xml)
 * using the HALBuilder {@link Representation}
 * @author patrickvk
 *
 */
public class HalHttpMessageConverter extends AbstractHttpMessageConverter<Object> {

	// Supported character set
	public static final Charset CHARSET = Charset.forName("UTF-8");

	// MediaTypes
	public static MediaType HAL_JSON = new MediaType("application", "hal+json", CHARSET);
	public static MediaType HAL_XML = new MediaType("application", "hal+xml", CHARSET);

	// RepresentationFactory to read resources
	private final RepresentationFactory factory;
	
	// Modules registered with this converter
	private final List<HalHttpMessageConverterModule> modules = new ArrayList<HalHttpMessageConverterModule>();

	public HalHttpMessageConverter(RepresentationFactory factory) {
		super(HAL_JSON, HAL_XML);
		this.factory = factory;
	}

	/* (non-Javadoc)
	 * @see org.springframework.http.converter.AbstractHttpMessageConverter#canRead(java.lang.Class, org.springframework.http.MediaType)
	 */
	@Override
	public boolean canRead(java.lang.Class<?> type, MediaType mediaType) {
		if (!canRead(mediaType)) {
			return false;
		}
		if (ReadableRepresentation.class.isAssignableFrom(type)) {
			return true;
		} else if (hasReaderModule(type)) {
			return true;
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see org.springframework.http.converter.AbstractHttpMessageConverter#canWrite(java.lang.Class, org.springframework.http.MediaType)
	 */
	@Override
	public boolean canWrite(Class<?> type, MediaType mediaType) {
		if (!canWrite(mediaType)) {
			return false;
		}
		if (ReadableRepresentation.class.isAssignableFrom(type)) {
			return true;
		} else if (hasWriterModule(type)) {
			return true;
		}
		return false;
	}

	/* (non-Javadoc)
	 * @see org.springframework.http.converter.AbstractHttpMessageConverter#readInternal(java.lang.Class, org.springframework.http.HttpInputMessage)
	 */
	@Override
	protected Object readInternal(Class<? extends Object> type,
			HttpInputMessage message) throws IOException,
			HttpMessageNotReadableException {
		ReadableRepresentation representation = factory.readRepresentation(new InputStreamReader(message.getBody()));
		if (ReadableRepresentation.class.isAssignableFrom(type)) {
			return representation;
		} else if (hasReaderModule(type)) {
			return getReaderModule(type).read(representation, type);
		}
		throw new IllegalArgumentException(String.format("Cannot read class %s", type));
	}

	/* (non-Javadoc)
	 * @see org.springframework.http.converter.AbstractHttpMessageConverter#supports(java.lang.Class)
	 */
	@Override
	protected boolean supports(Class<?> type) {
		throw new UnsupportedOperationException();
	}

	/* (non-Javadoc)
	 * @see org.springframework.http.converter.AbstractHttpMessageConverter#writeInternal(java.lang.Object, org.springframework.http.HttpOutputMessage)
	 */
	@Override
	protected void writeInternal(Object target, HttpOutputMessage outputMessage)
			throws IOException, HttpMessageNotWritableException {
		ReadableRepresentation rep = getRepresentation(target);
		MediaType contentType = outputMessage.getHeaders().getContentType();
		String mediaType = contentType.getType() + "/" + contentType.getSubtype();
		Writer writer = new OutputStreamWriter(outputMessage.getBody());
		rep.toString(mediaType, writer);
	}

	/**
	 * Retrieve {@link ReadableRepresentation} from
	 * the supplied target
	 * @param target
	 * @return
	 */
	private ReadableRepresentation getRepresentation(Object target) {
		if (ReadableRepresentation.class.isAssignableFrom(target.getClass())) {
			return (ReadableRepresentation)target;
		} else if (hasWriterModule(target.getClass())) {
			return getWriterModule(target.getClass()).write(target, this.factory);
		}
		throw new IllegalArgumentException(String.format("Unable to create Representation from object of type %s", target.getClass()));
	}
	
	/**
	 * Is there a {@link HalHttpMessageConverterModule}
	 * registered that can read this {@link Class}
	 * @param type
	 * @return true if there is, false if there ain't
	 */
	private boolean hasReaderModule(Class<?> type) {
		HalHttpMessageConverterModule module = getReaderModule(type);
		return module != null ? true : false;
	}
	
	/**
	 * Is there a {@link HalHttpMessageConverterModule}
	 * registered that can write this {@link Class}
	 * @param type
	 * @return true if there is, false if there ain't
	 */
	private boolean hasWriterModule(Class<?> type) {
		HalHttpMessageConverterModule module = getWriterModule(type);
		return module != null ? true : false;
	}
	
	/**
	 * Get a {@link HalHttpMessageConverterModule}
	 * that can read the supplied {@link Class} type, 
	 * if one exists.
	 *
	 * @param type {@link Class} to read
	 * @return matching {@link HalHttpMessageConverterModule} module or null
	 */
	private HalHttpMessageConverterModule getReaderModule(Class<?> type) {
		for (HalHttpMessageConverterModule module : this.modules) {
			if (module.canRead(type)) {
				return module;
			}
		}
		return null;
	}
	
	/**
	 * Get a {@link HalHttpMessageConverterModule}
	 * that can write the supplied {@link Class}
	 * or null
	 * @param type {@link Class} to write
	 * @return a {@link HalHttpMessageConverterModule} or null
	 */
	private HalHttpMessageConverterModule getWriterModule(Class<?> type) {
		for (HalHttpMessageConverterModule module : this.modules) {
			if (module.canWrite(type)) {
				return module;
			}
		}
		return null;
	}
	
	/**
	 * Add a {@link HalHttpMessageConverterModule}
	 * to handle a conversion
	 * @param module
	 */
	public void addModule(HalHttpMessageConverterModule module) {
		this.modules.add(module);
	}
	
	/**
	 * Get the {@link HalHttpMessageConverterModule} 
	 * modules registered with this converter.
	 * @return
	 */
	public List<HalHttpMessageConverterModule> getModules() {
		return Collections.unmodifiableList(this.modules);
	}

}
