package net.ericaro.neobin.plugin;

import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashSet;

import net.ericaro.neobin.NeoBin;
import net.ericaro.neobin.NeoBinException;

import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.project.MavenProject;

import edu.uci.ics.jung.graph.Graph;

/** generate a NeoBin Mojo
 * 
 * 
 * @author eric
 * @goal generate
 * @phase generate-sources
 * @since 2.0
 */
public class NeoBinMojo extends AbstractMojo {

	/**
	 * The current Maven project.
	 * 
	 * @parameter default-value="${project}"
	 * @readonly
	 * @required
	 */
	private MavenProject project;

	/**
	 * The set of compile source roots whose contents are not generated as part of the build, i.e. those that usually
	 * reside somewhere below "${basedir}/src" in the project structure. Files in these source roots are owned by the
	 * user and must not be overwritten with generated files.
	 */
	private Collection nonGeneratedSourceRoots;

	 /**
     * The directory where the NeoBin files (<code>*.xml</code>) are located.
     * 
     * @parameter expression="${sourceDirectory}" default-value="${basedir}/src/main/neobin"
     */
    private File sourceDirectory;

    /**
     * The directory where the files generated by NeoBin will be stored. The directory will be registered as a
     * compile source root of the project such that the generated files will participate in later build phases like
     * compiling and packaging.
     * 
     * @parameter expression="${outputDirectory}" default-value="${project.build.directory}/generated-sources/neobin"
     */
    private File outputDirectory;

    /**
     * The granularity in milliseconds of the last modification date for testing whether a source needs recompilation.
     * 
     * @parameter expression="${lastModGranularityMs}" default-value="0"
     */
    private int staleMillis;
    
    /**
     * Verbose mode
     * 
     * @parameter expression="${debug}" default-value="false"
     */
    private boolean debug;
    

    /**
     * A set of Ant-like inclusion patterns used to select files from the source directory for processing. By default,
     * the patterns <code>**&#47;*.xml</code>  are used to select neobin files.
     * 
     * @parameter
     */
    private String[] includes;

    /**
     * A set of Ant-like exclusion patterns used to prevent certain files from being processed. By default, this set is
     * empty such that no files are excluded.
     * 
     * @parameter
     */
    private String[] excludes;

    /**
     * {@inheritDoc}
     */
    protected File getSourceDirectory()
    {
        return this.sourceDirectory;
    }

    /**
     * {@inheritDoc}
     */
    protected String[] getIncludes()
    {
        if ( this.includes != null )
        {
            return this.includes;
        }
        else
        {
            return new String[] { "**/*.xml", "**/*.dzl" };
        }
    }

    /**
     * {@inheritDoc}
     */
    protected String[] getExcludes()
    {
        return this.excludes;
    }

    /**
     * {@inheritDoc}
     */
    protected File getOutputDirectory()
    {
        return this.outputDirectory;
    }

    /**
     * {@inheritDoc}
     */
    protected int getStaleMillis()
    {
        return this.staleMillis;
    }

    /**
     * {@inheritDoc}
     */
    protected File[] getCompileSourceRoots()
    {
        return new File[] { getOutputDirectory() };
    }

	
	/**
	 * Execute the tool.
	 * 
	 * @throws MojoExecutionException
	 *             If the invocation of the tool failed.
	 * @throws MojoFailureException
	 *             If the tool reported a non-zero exit code.
	 */
	public void execute() throws MojoExecutionException, MojoFailureException {
		
		
		File[] neobins = scanForFiles();
		getLog().info("starting NeoBin compilation from: " + getSourceDirectory());

		if (neobins == null) {
			getLog().info("Skipping non-existing neobin in directory: " + getSourceDirectory());
			return;
		} else if (neobins.length <= 0) {
			getLog().info("Skipping - all neobin files are up to date");
		} else {
			determineNonGeneratedSourceRoots();
			processNeobin(neobins);
			getLog().info("Processed " + neobins.length + " neobin " + (neobins.length != 1 ? "s" : ""));
		}

		File[] compileSourceRoots = getCompileSourceRoots();
		for (int i = 0; i < compileSourceRoots.length; i++) 
			addSourceRoot(compileSourceRoots[i]);
		
	}

	/**
	 * Passes the specified grammar file through the tool.
	 * 
	 * @param neobin
	 *            The file containing the neobin file to process, must not be <code>null</code>.
	 * @throws MojoExecutionException
	 *             If the invocation of the tool failed.
	 * @throws MojoFailureException
	 *             If the tool reported a non-zero exit code.
	 */
	protected void processNeobin(File[] neobin) throws MojoExecutionException, MojoFailureException{
			try {
				NeoBin.generate(outputDirectory, neobin);
			} catch (NeoBinException e) {
				throw new MojoFailureException(e.getMessage());
			}
	}

	/**
	 * Scans the configured source directory for files which need processing.
	 * 
	 * @return An array of files containing the found grammar files or <code>null</code> if the source
	 *         directory does not exist.
	 * @throws MojoExecutionException
	 *             If the source directory could not be scanned.
	 */
	private File[] scanForFiles() throws MojoExecutionException {
		if (!getSourceDirectory().isDirectory()) {
			return null;
		}

		getLog().debug("Scanning for grammars: " + getSourceDirectory());
		try {
			NeoBinDirectoryScanner scanner = new NeoBinDirectoryScanner();
			scanner.setSourceDirectory(getSourceDirectory());
			scanner.setIncludes(getIncludes());
			scanner.setExcludes(getExcludes());
			scanner.setOutputDirectory(getOutputDirectory());
			scanner.setStaleMillis(getStaleMillis());
			scanner.scan();
			File[] files = scanner.getIncludedFiles();
			getLog().debug("Found files: " + Arrays.asList(files));
			return files;

		} catch (Exception e) {
			throw new MojoExecutionException("Failed to scan for grammars: " + getSourceDirectory(), e);
		}
	}

	/**
	 * Determines those compile source roots of the project that do not reside below the project's build directories.
	 * These compile source roots are assumed to contain hand-crafted sources that must not be overwritten with
	 * generated files. In most cases, this is simply "${project.build.sourceDirectory}".
	 * 
	 * @throws MojoExecutionException
	 *             If the compile source rotos could not be determined.
	 */
	private void determineNonGeneratedSourceRoots() throws MojoExecutionException {
		this.nonGeneratedSourceRoots = new LinkedHashSet();
		try {
			String targetPrefix = new File(this.project.getBuild().getDirectory()).getCanonicalPath() + File.separator;
			Collection sourceRoots = this.project.getCompileSourceRoots();
			for (Iterator it = sourceRoots.iterator(); it.hasNext();) {
				File sourceRoot = new File(it.next().toString());
				if (!sourceRoot.isAbsolute()) {
					sourceRoot = new File(this.project.getBasedir(), sourceRoot.getPath());
				}
				String sourcePath = sourceRoot.getCanonicalPath();
				if (!sourcePath.startsWith(targetPrefix)) {
					this.nonGeneratedSourceRoots.add(sourceRoot);
					getLog().debug("Non-generated compile source root: " + sourceRoot);
				} else {
					getLog().debug("Generated compile source root: " + sourceRoot);
				}
			}
		} catch (IOException e) {
			throw new MojoExecutionException("Failed to determine non-generated source roots", e);
		}
	}

	/**
	 * Determines whether the specified directory denotes a compile source root of the current project.
	 * 
	 * @param directory
	 *            The directory to check, must not be <code>null</code>.
	 * @return <code>true</code> if the specified directory is a compile source root of the project, <code>false</code> otherwise.
	 */
	protected boolean isSourceRoot(File directory) {
		return this.nonGeneratedSourceRoots.contains(directory);
	}

	/**
	 * Registers the specified directory as a compile source root for the current project.
	 * 
	 * @param directory
	 *            The absolute path to the source root, must not be <code>null</code>.
	 */
	private void addSourceRoot(File directory) {
		if (this.project != null) {
			getLog().debug("Adding compile source root: " + directory);
			this.project.addCompileSourceRoot(directory.getAbsolutePath());
		}
	}
}
