package net.ericaro.neobin;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import net.ericaro.neobin.v1.DataType;
import net.ericaro.neobin.v1.ManyType;
import net.ericaro.neobin.v1.Neobin;
import net.ericaro.neobin.v1.Neobin.Transitions.Transition.Var;
import net.ericaro.neogrex.RegExpGraphBuilder;
import net.ericaro.parser.RegExpParser;

public class BinaryFormatBuilder {

	private Neobin src;
	private BinGrex grex;

	private String buildType(DataType type) {
		switch (type) {
		case FLOAT:
			return "float";
		case STRING:
			return "java.lang.String";
		case INT:
			return "int";
		case LONG:
			return "long";
		default:
			return null;
		}
	}

	private Transition buildTransition(net.ericaro.neobin.v1.Neobin.Transitions.Transition t) {

		return new Transition(t.getName(), buildBinaryTypes(t.getVar()), null);

	}

	private List<BinaryType> buildBinaryTypes(List<Var> var) {
		List<BinaryType> res = new ArrayList<BinaryType>();
		for (Var v : var) {
			boolean many = v.getMany().equals(ManyType.MANY);
			PrimitiveType p = buildBinaryType(v);
			
			BinaryType bt = (many?p.getBuffer(): p.getType());
			res.add( new BinaryType(bt, v.getName()) );
		}
		return res;
	}

	private PrimitiveType buildBinaryType(Var v) {
		switch (v.getType()) {
		case BYTE:
			return PrimitiveType.BYTE;
		case INT:
			return PrimitiveType.INT;
		case LONG:
			return PrimitiveType.LONG;
		case FLOAT:
			return PrimitiveType.FLOAT;
		case DOUBLE:
			return PrimitiveType.DOUBLE;
		case STRING:
			return PrimitiveType.STRING;
		default:
			throw new IllegalArgumentException("unkown type");
		}
	}

	/**
	 * build a jung graph of states and transitions based on the xml description
	 * 
	 * @param expression
	 * @return
	 */
	private void buildGraph() {

		// build a faster index for transition by name
		final HashMap<String, Transition> index = new HashMap<String, Transition>();
		for (net.ericaro.neobin.v1.Neobin.Transitions.Transition t : src.getTransitions().getTransition())
			index.put(t.getName(), buildTransition(t));

		try {
			grex = RegExpParser.parse(src.getExpression(), new RegExpGraphBuilder<BinGrex, State, Transition>() {
				public BinGrex terminal(String name) {
					return new BinGrex( index.get(name), new State(), new State() );
				}
			});
		} catch (net.ericaro.parser.ParseException e) {
			throw new RuntimeException("failed to generate the Graph for expression "+ src.getExpression(), e);
		}
		
		
		{
			// check if transitions are the sames
			Set<String> gTransitions = new HashSet<String>(); // transitions
																// from the
																// graph
			for (Transition t : grex.getTransitions())
				gTransitions.add(t.getMethod() );
			

			// transitions from the XML definition
			Set<String> xTransitions = new HashSet<String>(index.keySet());
			if (!xTransitions.containsAll(gTransitions)) {
				// removed known transition, leaving only the undefined ones
				gTransitions.removeAll(xTransitions);
				throw new UndefinedTransitionException(gTransitions);
			}

			if (!gTransitions.containsAll(xTransitions)) {
				xTransitions.removeAll(gTransitions); // xtransition contains "extra transitions
				// TODO issue a warning here
			}

		}

		// generate state name and id
		int i = 0;
		for (State state : grex.getStates()) {
			state.setName(src.getName() + (++i));
			state.setId(i);
		}
		
		
	}

	/**
	 * parse the graph, and set a unique name for each state.
	 * 
	 */
	private void buildStateNames() {

		grex.getStartState().setName(src.getName());

		for (net.ericaro.neobin.v1.Neobin.States.State s : src.getStates().getState())
			getStateByPath(s.getPath()).setName(s.getValue());
	}

	private State getStateByPath(String key) {
		String[] elements = key.split("\\.");
		State current = grex.getStartState();
		for (String next : elements) {
			for (Transition t : grex.getOutEdges(current)) {
				if (next.equals(t.getMethod())) {
					current = grex.getDest(t);
					break; // get out of the transition loop
				}
			}
		}
		// now the whole path has been parsed, and current is our goal
		return current;
	}

	protected void buildConnectedGraph() {
		// now loop the graph to connect transitions to their child
		for (Transition t : grex.getTransitions()) {
			State s = grex.getSource(t);
			State dest = grex.getDest(t);
			
			s.getTransitions().add(t);
			t.setNextState(dest);
		}
		
		for (State state : grex.getStates()) {
			int i = 0;
			for (Transition t : state.getTransitions())
				t.setId(i++);
		}
	}

	
	/** Main builder method
	 * 
	 * @param src
	 * @return
	 */
	public BinaryFormat build(Neobin src) {
		this.src = src;
		buildGraph(); // compile the expression and build the grex object
		// try to rename states based on the xml description 
		buildStateNames();
		// fully connect transition to state, and state to transition, to ease the generation templates
		buildConnectedGraph();
		
		// states to be generated
		ArrayList<State> states = new ArrayList<State>();
		// do not append the start edge (because it
		for(State s: grex.getStates())
			if ( grex.getStartState() != s && ! s.isTerminal() )
				states.add(s);
		
		BinaryFormat bf = new BinaryFormat(src.getPackage(), grex.getStartState(), states, grex.getTransitions() );

		return bf;
	}

}
