package net.emaze.dysfunctional;

import java.util.Iterator;
import net.emaze.dysfunctional.contracts.dbc;
import net.emaze.dysfunctional.dispatching.adapting.IgnoreParameter;
import net.emaze.dysfunctional.dispatching.delegates.Delegate;
import net.emaze.dysfunctional.dispatching.delegates.Provider;
import net.emaze.dysfunctional.filtering.DropWhile;
import net.emaze.dysfunctional.filtering.FilteringIterator;
import net.emaze.dysfunctional.filtering.UntilCount;
import net.emaze.dysfunctional.iterations.*;
import net.emaze.dysfunctional.options.Maybe;

/**
 * Iterators and Iterables shortcuts.
 *
 * @author rferranti
 */
public abstract class Iterations {

    /**
     * Creates an iterable usable only ONE TIME from an iterator.
     *
     * @param <T> the iterator element type parameter
     * @param iterator the iterator to be yielded by the iterable
     * @return an iterable consumable only once
     */
    public static <T> Iterable<T> oneTime(Iterator<T> iterator) {
        return new OneTimeIterable<T>(iterator);
    }

    /**
     * Creates an empty iterator
     *
     * @param <T> the element parameter type
     * @return an empty iterator.
     */
    public static <T> Iterator<T> iterator() {
        return new EmptyIterator<T>();
    }

    /**
     * Creates an iterator from the passed value.
     *
     * @param <T> the element parameter type
     * @param value the value to be yielded by the iterator
     * @return an iterator.
     */
    public static <T> Iterator<T> iterator(T value) {
        return new SingletonIterator<T>(value);
    }

    /**
     * Creates an iterator from the passed values.
     *
     * @param <T> the elements parameter type
     * @param first the first element
     * @param second the second element
     * @return an iterator.
     */
    public static <T> Iterator<T> iterator(T first, T second) {
        return ArrayIterator.of(first, second);
    }

    /**
     * Creates an iterator from the passed values.
     *
     * @param <T> the elements parameter type
     * @param first the first element
     * @param second the second element
     * @param third the third element
     * @return an iterator.
     */
    public static <T> Iterator<T> iterator(T first, T second, T third) {
        return ArrayIterator.of(first, second, third);
    }

    /**
     * Creates an iterator from the passed array.
     *
     * @param <T> the array element parameter type
     * @param values the values to be yielded by the iterator
     * @return an iterator.
     */
    public static <T> Iterator<T> iterator(T... values) {
        return new ArrayIterator<T>(values);
    }

    /**
     * An iterator generated by a {@code Provider} providing the next item or nothing when
     * the iteration should be completed.
     *
     * @param <T> the elements parameter type
     * @param generator the generator of the values to be yielded by the iterator
     * @return an iterator
     */
    public static <T> Iterator<T> iterator(Provider<Maybe<T>> generator) {
        dbc.precondition(generator != null, "generator cannot be null");
        final IteratorGenerator<T> iterator = new IteratorGenerator<T>(null, new IgnoreParameter<Maybe<T>, T>(generator));
        return new FilteringIterator<T>(iterator, new DropWhile<T>(new UntilCount<T>(1)));
    }

    /**
     * An iterator generated by iterative application of the {@code generator} function 
     * providing the next item or nothing when the iteration should be completed.
     *
     * The seed is the first item of the sequence, then each item is generated
     * calling the function with the previous generated item, producing the sequence
     * {@code seed, f(seed), f(f(seed)), ...}.
     *
     * @param <T> the elements parameter type
     * @param seed the initial input element passed to the generator function
     * @param generator the generator of the values to be yielded by the iterator
     * @return an iterator
     */
    public static <T> Iterator<T> iterator(T seed, Delegate<Maybe<T>, T> generator) {
        return new IteratorGenerator<T>(seed, generator);
    }

    /**
     * Creates an empty iterable.
     *
     * @param <T> the element parameter type
     * @return an iterable always yielding an empty iterator.
     */
    public static <T> Iterable<T> iterable() {
        return new EmptyIterable<T>();
    }

    /**
     * Creates an iterable from the passed value.
     *
     * @param <T> the element parameter type
     * @param value the value to be yielded by the iterator
     * @return an iterable.
     */
    public static <T> Iterable<T> iterable(T value) {
        return new SingletonIterable<T>(value);
    }

    /**
     * Creates an iterable from the passed values.
     *
     * @param <T> the elements parameter type
     * @param first the first element
     * @param second the second element
     * @return an iterable.
     */
    public static <T> Iterable<T> iterable(T first, T second) {
        return ArrayIterable.of(first, second);
    }

    /**
     * Creates an iterable from the passed values.
     *
     * @param <T> the elements parameter type
     * @param first the first element
     * @param second the second element
     * @param third the third element
     * @return an iterable.
     */
    public static <T> Iterable<T> iterable(T first, T second, T third) {
        return ArrayIterable.of(first, second, third);
    }

    /**
     * Creates an iterable from the passed array.
     *
     * @param <T> the array element parameter type
     * @param values the values to be yielded by the iterator
     * @return an iterable.
     */
    public static <T> Iterable<T> iterable(T... values) {
        return new ArrayIterable<T>(values);
    }
}
