package net.emaze.dysfunctional.iterations;

import java.util.NoSuchElementException;
import net.emaze.dysfunctional.contracts.dbc;
import net.emaze.dysfunctional.dispatching.delegates.Delegate;
import net.emaze.dysfunctional.options.Box;
import net.emaze.dysfunctional.options.Maybe;

/**
 * An iterator generated by iterative application of a function providing the next item
 * or nothing when the iteration should be completed.
 *
 * The seed is the first item of the sequence, then each item is generated
 * calling the function with the previous generated item, producing the sequence
 * {@code seed, f(seed), f(f(seed)), ...}.
 */
public class IteratorGenerator<T> extends ReadOnlyIterator<T> {

    private final Delegate<Maybe<T>, T> generator;
    private final Box<T> next;
    private T current;
    private boolean active = true;

    public IteratorGenerator(T seed, Delegate<Maybe<T>, T> generator) {
        dbc.precondition(generator != null, "generator cannot be null");
        this.generator = generator;
        this.next = Box.of(seed);
        this.current = seed;
    }

    @Override
    public boolean hasNext() {
        return fetchNext();
    }

    @Override
    public T next() {
        if (!fetchNext()) {
            throw new NoSuchElementException("generation has been completed");
        }
        return next.unload().value();
    }

    private boolean fetchNext() {
        if (active && next.isEmpty()) {
            final Maybe<T> got = generator.perform(current);
            active = got.hasValue();
            if (active) {
                current = got.value();
                next.setContent(current);
            }
        }
        return active;
    }
}
