package net.eledge.urlbuilder;

import net.eledge.urlbuilder.exception.InvalidUrlException;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.math.NumberUtils;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

public class UrlBuilder {

    public static final String PATH_SEPERATOR = "/";

    private String protocol = "http";
    private String domain = null;
    private int port = 80;
    private StringBuilder path;
    private String anchor = null;

    private boolean disableProtocol = false;
    private boolean disableDomain = false;
    private boolean disableEncoding = false;
    private boolean trailingSlash = false;

    private Map<String, List<String>> paramMap = new LinkedHashMap<String, List<String>>();

    /**
     * Create an instance of UrlBuilder with a base url to start with.
     *
     * @param url Base url to start with, can be relative.
     */
    public UrlBuilder(String url) {
        setBaseUrl(StringUtils.replace(url, "&amp;", "&"));
    }

    /**
     * Disable protocol inclusion for toString().
     * Is irrelevant if no domain is displayed.
     *
     * @return instance of UrlBuilder
     */
    public UrlBuilder disableProtocol() {
        disableProtocol = true;
        return this;
    }

    /**
     * Don't output domain on toString().
     * When disabled, protocol will be disabled as well.
     *
     * @return instance of UrlBuilder
     */
    public UrlBuilder disableDomain() {
        disableDomain = true;
        return this;
    }

    /**
     * Disable default encoding on parameters.
     *
     * @return instance of UrlBuilder
     */
    public UrlBuilder disableEncoding() {
        disableEncoding = true;
        return this;
    }

    /**
     * If url ends with a path instead of page by default a / will be added.
     * Use this method to disable that trailing slash
     *
     * @return instance of UrlBuilder
     */
    public UrlBuilder disableTrailingSlash() {
        trailingSlash = false;
        return this;
    }

    /**
     * Add an anchor to url.
     *
     * @param anchor the anchor string to use (excluding the dash)
     * @return instance of UrlBuilder
     */
    public UrlBuilder setAnchor(String anchor) {
        this.anchor = anchor;
        return this;
    }

    /**
     * Add one or more path segments to url.
     *
     * @param paths One or more path segments
     * @return instance of UrlBuilder
     */
    public UrlBuilder addPath(String... paths) {
        if (StringUtils.isNoneBlank(paths)) {
            for (String p : paths) {
                p = StringUtils.strip(p, "/?&");
                path.append(PATH_SEPERATOR).append(p);
            }
            trailingSlash = true;
        }
        return this;
    }

    /**
     * Add page (including extension) to path
     *
     * @param page complete page name ie: page.html
     * @return instance of UrlBuilder
     */
    public UrlBuilder addPage(String page) {
        if (StringUtils.isNotBlank(page)) {
            page = StringUtils.strip(page, "/?&");
            path.append(PATH_SEPERATOR).append(page);
            trailingSlash = false;
        }
        return this;
    }

    /**
     * Extract parameters from existing url and add them.
     *
     * @param url The url to extract params from
     * @param ignoreKeys One or more param keys to ignore
     * @return instance of UrlBuilder
     */
    public UrlBuilder addParamsFromURL(String url, String... ignoreKeys) {
        if (StringUtils.isNotBlank(url)) {
            String[] parameters = StringUtils.split(url, "&");
            for (String string : parameters) {
                String[] param = StringUtils.split(string, "=");
                if (param.length == 2) {
                    if (!StringUtils.isNoneBlank(ignoreKeys)
                            || !ArrayUtils.contains(ignoreKeys, param[0])) {
                        addParam(param[0], true, param[1]);
                    }
                }
            }
        }
        return this;
    }

    /**
     * Add number parameter to url
     *
     * @param key The key of the parameter
     * @param value The value of the parameter
     * @return instance of UrlBuilder
     */
    public UrlBuilder addParam(String key, long value) {
        return addParam(key, false, String.valueOf(value));
    }

    /**
     * Add parameter with one or more values
     *
     * @param key The key of the parameter
     * @param values One or more value(s) of the parameter
     * @return instance of UrlBuilder
     */
    public UrlBuilder addParam(String key, String... values) {
        return addParam(key, false, values);
    }

    /**
     * Add parameter with one or more values
     *
     * @param key The key of the parameter
     * @param override Override existing parameters
     * @param values One or more value(s) of the parameter
     * @return instance of UrlBuilder
     */
    private UrlBuilder addParam(String key, boolean override, String... values) {
        if (StringUtils.isNotBlank(key) && StringUtils.isNoneBlank(values)) {
            List<String> list = null;
            if (!override && paramMap.containsKey(key)) {
                list = paramMap.get(key);
            } else {
                list = new ArrayList<String>();
                paramMap.put(key, list);
            }
            for (String value : values) {
                if (!list.contains(value)) {
                    list.add(value);
                }
            }
        }
        return this;
    }

    /**
     * @param key The key of the parameter
     * @param value The value of the parameter
     * @return instance of UrlBuilder
     */
    public UrlBuilder overrideParam(String key, long value) {
        return addParam(key, true, String.valueOf(value));
    }

    /**
     * @param key The key of the parameter
     * @param values One or more value(s) of the parameter
     * @return instance of UrlBuilder
     */
    public UrlBuilder overrideParam(String key, String... values) {
        return addParam(key, true, values);
    }

    /**
     * @param key The key of the parameter
     * @return instance of UrlBuilder
     */
    public boolean hasParam(String key) {
        return paramMap.containsKey(key);
    }

    /**
     * @param key The key of the parameter
     * @return instance of UrlBuilder
     */
    public UrlBuilder removeParam(String key) {
        if (StringUtils.isNotBlank(key)) {
            if (paramMap.containsKey(key)) {
                paramMap.remove(key);
            }
        }
        return this;
    }

    /**
     * Remove the value of a (multiple) parameter if present
     *
     * @param key The key of the parameter
     * @param value The value to remove if present
     * @return instance of UrlBuilder
     */
    public UrlBuilder removeParamDefault(String key, String value) {
        if (StringUtils.isNotBlank(key)) {
            if (paramMap.containsKey(key)) {
                if (paramMap.get(key).contains(value)) {
                    paramMap.get(key).remove(value);
                }
            }
        }
        return this;
    }

    /**
     * Remove value(s) that start with a given string of a (multiple) parameter if present
     *
     * @param key The key of the parameter
     * @param value The value to remove if present
     * @return instance of UrlBuilder
     */
    public UrlBuilder removeParamValuesStartWith(String key, String value) {
        if (paramMap.containsKey(key)) {
            List<String> toRemove = new ArrayList<String>();
            for (String string : paramMap.get(key)) {
                if (StringUtils.startsWith(string, value)) {
                    toRemove.add(string);
                }
            }
            if (toRemove.size() > 0) {
                paramMap.get(key).removeAll(toRemove);
            }
            if (paramMap.get(key).size() == 0) {
                removeParam(key);
            }
        }
        return this;
    }

    /**
     * Set the domain of the url
     *
     * @param newDomain The domain to set.
     * @return instance of UrlBuilder
     */
    public UrlBuilder setDomain(String newDomain) {
        if (StringUtils.isNotBlank(newDomain)) {
            if (StringUtils.contains(newDomain, "://")) {
                newDomain = StringUtils.substringAfter(newDomain, "://");
            }
            domain = StringUtils.substringBefore(newDomain, PATH_SEPERATOR);
        }
        return this;
    }

    /**
     * Generates the canonical url.
     *
     * @return The canonical url as string
     * @throws InvalidUrlException When no domain is preset.
     */
    public String toCanonicalUrl() throws InvalidUrlException {
        if (StringUtils.isBlank(domain)) {
            throw new InvalidUrlException("A Canonical URL should contain domain and protocol");
        }
        return createCanonical(true, true).toString();
    }

    /**
     * Generates the url.
     *
     * @return The complete url as a string
     */
    @Override
    public String toString() {
        StringBuilder sb = createCanonical(!disableProtocol, !disableDomain);
        if (!paramMap.isEmpty()) {
            boolean first = true;
            sb.append("?");
            for (Entry<String, List<String>> entry : paramMap.entrySet()) {
                for (String value : entry.getValue()) {
                    if (!first) {
                        sb.append("&");
                    }
                    sb.append(entry.getKey()).append("=").append(encode(value));
                    first = false;
                }
            }
        }
        if (StringUtils.isNotBlank(anchor)) {
            sb.append("#").append(anchor);
        }
        return sb.toString();
    }

    private void setBaseUrl(String url) {
        url = StringUtils.stripEnd(url, "/?&");
        if (StringUtils.isBlank(url) || StringUtils.startsWith(url, PATH_SEPERATOR)) {
            disableDomain = true;
        } else {
            if (StringUtils.contains(url, "://")) {
                protocol = StringUtils.substringBefore(url, "://");
                url = StringUtils.substringAfter(url, "://");
            }
            domain = StringUtils.substringBefore(url, PATH_SEPERATOR);
            if (!StringUtils.contains(domain, ":")) {
                url = StringUtils.substringAfter(url, domain);
            } else {
                port = NumberUtils.toInt(StringUtils.substringAfter(domain, ":"));
                domain = StringUtils.substringBefore(domain, ":");
                url = StringUtils.substringAfter(url, ":" + port);
            }
        }
        if (StringUtils.contains(url, "#")) {
            anchor = StringUtils.substringAfterLast(url, "#");
            url = StringUtils.substringBeforeLast(url, "#");
        }
        if (StringUtils.contains(url, '?')) {
            url = stripBaseUrl(url);
        }
        path = new StringBuilder(StringUtils.replace(url, "//", PATH_SEPERATOR));
    }

    private String stripBaseUrl(String url) {
        String[] result = StringUtils.split(url, '?');
        String stripped;
        String toProcess = null;
        if (result.length == 2) {
            stripped = result[0];
            toProcess = result[1];
        } else {
            if (StringUtils.endsWith(url, "?")) {
                stripped = result[0];
            } else {
                stripped = "";
                toProcess = result[0];
            }
        }
        addParamsFromURL(toProcess);
        return stripped;
    }

    private StringBuilder createCanonical(boolean addProtocol, boolean addDomain) {
        StringBuilder sb = new StringBuilder();
        if (addDomain && StringUtils.isNotBlank(domain)) {
            if (addProtocol) {
                sb.append(protocol).append("://");
            }
            sb.append(domain);
            if (port != 80) {
                sb.append(":").append(port);
            }
        }
        if (path != null) {
            sb.append(path.toString());
        }
        if (trailingSlash) {
            sb.append(PATH_SEPERATOR);
        }
        return sb;
    }

    private String encode(String value) {
        if (!disableEncoding) {
            try {
                value = URLEncoder.encode(value, "UTF-8");
            } catch (UnsupportedEncodingException e) {
                // should never happen...
            }
        }
        return value;
    }
}
