package net.eksb.kswingutil.ext

import net.eksb.kswingutil.property.MutableProperty
import net.eksb.kswingutil.property.Property
import java.awt.Component

/**
 * One-way bind a [java.awt.Component] to a [Property].
 * This component's property is immediately set to the property's value.
 * Then changes to the property are propagated to this component.
 *
 * @param property Property to bind from
 * @param setter lambda to set this component's value.
 */
fun <T> Component.bind(property: Property<T>, setter:(T)->Unit ) {
	val listener = { _:T, new:T -> setter(new) }
	property.addListener(listener)
	setter(property.value)
	onDispose { property.removeListener(listener) }
}

/**
 * A change listener that only gives the new value.
 */
typealias NewValueListener<T> = (T)->Unit

/**
 * Run only one block, even if the block calls other blocks recursively.
 * Not thread safe. Presumably only called from the swing event dispatch thread.
 */
class RunOne {
	private var isRunning = false
	fun run(block:()->Unit) {
		if (! isRunning) {
			try {
				isRunning = true
				block.invoke()
			} finally {
				isRunning = false
			}
		}
	}
}

/**
 * Two-way bind a [Component] to a [Property].
 * This component's property is immediately set to the property's value (via [setter]).
 * Then changes to either are propagated to the other.
 *
 * @param property Property to bind to
 * @param setter lambda to set this component value
 * @param listenerRegister lambda to register a [NewValueListener] with this component
 */
fun <T> Component.bind(
	property: MutableProperty<T>,
	setter:(T)->Unit,
	listenerRegister:(NewValueListener<T>)->Unit,
) {

	val runOne = RunOne() // avoid recursive loop

	// copy from component to property
	// register change listener with component
	val componentChangeListener = { value:T ->
		runOne.run {
			property.value = value
		}
	}
	listenerRegister.invoke(componentChangeListener)

	// copy from property to component
	fun propertyToComponent() {
		runOne.run {
			setter( property.value )
		}
	}
	val listener = { old:T, new:T -> propertyToComponent() }
	property.addListener(listener)
	onDispose {
		property.removeListener( listener )
	}

	// initialize component with the property's value
	propertyToComponent()
}