/*
 * Decompiled with CFR 0.152.
 */
package net.e175.klaus.zip;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.zip.ZipException;
import net.e175.klaus.zip.BinaryMapper;

public final class ZipPrefixer {
    private static final Logger LOG = Logger.getLogger(ZipPrefixer.class.getName());
    static final BinaryMapper.PatternSpec EOCDR = new BinaryMapper.PatternSpec(ByteOrder.LITTLE_ENDIAN, BinaryMapper.FieldSpec.of(4, "eocdrSignature", new byte[]{80, 75, 5, 6}), BinaryMapper.FieldSpec.of(2, "numberOfThisDisk"), BinaryMapper.FieldSpec.of(2, "numberOfStartDiskOfCD"), BinaryMapper.FieldSpec.of(2, "numberOfEntriesInCDonThisDisk"), BinaryMapper.FieldSpec.of(2, "totalNumberOfEntriesInCD"), BinaryMapper.FieldSpec.of(4, "sizeOfCD"), BinaryMapper.FieldSpec.of(4, "offsetOfStartOfCD"), BinaryMapper.FieldSpec.of(2, "commentLength"));
    static final BinaryMapper.PatternSpec CFH = new BinaryMapper.PatternSpec(ByteOrder.LITTLE_ENDIAN, BinaryMapper.FieldSpec.of(4, "centralFileHeader", new byte[]{80, 75, 1, 2}), BinaryMapper.FieldSpec.of(2, "versionMadeBy"), BinaryMapper.FieldSpec.of(2, "versionNeededToExtract"), BinaryMapper.FieldSpec.of(2, "generalPurposeBitFlag"), BinaryMapper.FieldSpec.of(2, "compressionMethod"), BinaryMapper.FieldSpec.of(2, "lastModFileTime"), BinaryMapper.FieldSpec.of(2, "lastModFileDate"), BinaryMapper.FieldSpec.of(4, "crc32"), BinaryMapper.FieldSpec.of(4, "compressedSize"), BinaryMapper.FieldSpec.of(4, "uncompressedSize"), BinaryMapper.FieldSpec.of(2, "fileNameLength"), BinaryMapper.FieldSpec.of(2, "extraFieldLength"), BinaryMapper.FieldSpec.of(2, "fileCommentLength"), BinaryMapper.FieldSpec.of(2, "diskNumberStart"), BinaryMapper.FieldSpec.of(2, "internalFileAttributes"), BinaryMapper.FieldSpec.of(4, "externalFileAttributes"), BinaryMapper.FieldSpec.of(4, "relativeOffsetOfLocalHeader"));
    static final BinaryMapper.PatternSpec LFH = new BinaryMapper.PatternSpec(ByteOrder.LITTLE_ENDIAN, BinaryMapper.FieldSpec.of(4, "centralFileHeader", new byte[]{80, 75, 3, 4}), BinaryMapper.FieldSpec.of(2, "versionNeededToExtract"), BinaryMapper.FieldSpec.of(2, "generalPurposeBitFlag"), BinaryMapper.FieldSpec.of(2, "compressionMethod"), BinaryMapper.FieldSpec.of(2, "lastModFileTime"), BinaryMapper.FieldSpec.of(2, "lastModFileDate"), BinaryMapper.FieldSpec.of(4, "crc32"), BinaryMapper.FieldSpec.of(4, "compressedSize"), BinaryMapper.FieldSpec.of(4, "uncompressedSize"), BinaryMapper.FieldSpec.of(2, "fileNameLength"), BinaryMapper.FieldSpec.of(2, "extraFieldLength"));
    static final BinaryMapper.PatternSpec ZIP64_EOCDL = new BinaryMapper.PatternSpec(ByteOrder.LITTLE_ENDIAN, BinaryMapper.FieldSpec.of(4, "zip64EOCDLSignature", new byte[]{80, 75, 6, 7}), BinaryMapper.FieldSpec.of(4, "numberOfDiskWithStartOfZip64EOCDL"), BinaryMapper.FieldSpec.of(8, "relativeOffsetOfZip64EOCDR"), BinaryMapper.FieldSpec.of(4, "totalNumberOfDisks"));
    static final BinaryMapper.PatternSpec ZIP64_EOCDR = new BinaryMapper.PatternSpec(ByteOrder.LITTLE_ENDIAN, BinaryMapper.FieldSpec.of(4, "zip64EOCDLSignature", new byte[]{80, 75, 6, 6}), BinaryMapper.FieldSpec.of(8, "sizeOfZip64eocdr"), BinaryMapper.FieldSpec.of(2, "versionMadeBy"), BinaryMapper.FieldSpec.of(2, "versionNeededToExtract"), BinaryMapper.FieldSpec.of(4, "numberOfThisDisk"), BinaryMapper.FieldSpec.of(4, "numberOfStartDiskOfCD"), BinaryMapper.FieldSpec.of(8, "numberOfEntriesInCDonThisDisk"), BinaryMapper.FieldSpec.of(8, "totalNumberOfEntriesInCD"), BinaryMapper.FieldSpec.of(8, "sizeOfCD"), BinaryMapper.FieldSpec.of(8, "offsetOfStartOfCD"));
    static final BinaryMapper.FieldSpec ZIP64_EIEF_SIGNATURE = BinaryMapper.FieldSpec.of(2, "zip64EIEFSignature", new byte[]{1, 0});

    private ZipPrefixer() {
    }

    public static long applyPrefixes(Path targetPath, byte[] ... prefixes) throws IOException {
        if (prefixes.length < 1) {
            return 0L;
        }
        Path original = targetPath.resolveSibling(targetPath.getFileName() + ".original");
        Files.move(targetPath, original, new CopyOption[0]);
        long prefixesLength = 0L;
        try (OutputStream out = Files.newOutputStream(targetPath, new OpenOption[0]);){
            for (byte[] prefix : prefixes) {
                out.write(prefix);
                prefixesLength += (long)prefix.length;
            }
            Files.copy(original, out);
        }
        Files.deleteIfExists(original);
        return prefixesLength;
    }

    public static long applyPrefixes(Path targetPath, List<Path> prefixFiles) throws IOException {
        if (prefixFiles.isEmpty()) {
            return 0L;
        }
        Path original = targetPath.resolveSibling(targetPath.getFileName() + ".original");
        Files.move(targetPath, original, new CopyOption[0]);
        long prefixesLength = 0L;
        try (OutputStream out = Files.newOutputStream(targetPath, new OpenOption[0]);){
            for (Path prefixFile : prefixFiles) {
                prefixesLength += Files.copy(prefixFile, out);
            }
            Files.copy(original, out);
        }
        Files.deleteIfExists(original);
        return prefixesLength;
    }

    public static void validateZipOffsets(Path targetPath) throws IOException {
        ZipPrefixer.adjustZipOffsets(targetPath, 0L);
    }

    public static void adjustZipOffsets(Path targetPath, long adjustment) throws IOException {
        Queue<BinaryMapper.Write> writeQueue;
        boolean mustAdjust = adjustment != 0L;
        try (SeekableByteChannel channel = Files.newByteChannel(targetPath, new OpenOption[0]);){
            writeQueue = ZipPrefixer.analyseOffsets(mustAdjust, adjustment, channel);
        }
        if (mustAdjust) {
            channel = Files.newByteChannel(targetPath, StandardOpenOption.WRITE);
            try {
                BinaryMapper.applyWrites(writeQueue, channel);
            }
            finally {
                if (channel != null) {
                    channel.close();
                }
            }
        }
    }

    private static Queue<BinaryMapper.Write> analyseOffsets(boolean mustAdjust, long adjustment, SeekableByteChannel channel) throws IOException {
        Optional<BinaryMapper.PatternInstance> zip64eocdlO;
        long numberOfCdEntries;
        Queue<BinaryMapper.Write> writeQueue = mustAdjust ? BinaryMapper.createWriteQueue() : null;
        BinaryMapper.PatternInstance eocdr = BinaryMapper.seek(EOCDR, channel, Long.MAX_VALUE, false).orElseThrow(() -> new ZipException("Unable to locate EOCDR. Probably not a ZIP file."));
        LOG.fine(() -> String.format("EOCDR found at offset: \"0x%08X\"", eocdr.position));
        boolean requiresZip64 = false;
        long cdOffset = eocdr.getUnsignedInt("offsetOfStartOfCD");
        if (cdOffset != 0xFFFFFFFFL) {
            if (mustAdjust) {
                writeQueue.add(eocdr.writeInt("offsetOfStartOfCD", (int)(cdOffset += adjustment)));
            }
        } else {
            requiresZip64 = true;
        }
        if ((numberOfCdEntries = (long)eocdr.getUnsignedShort("numberOfEntriesInCDonThisDisk")) == 65535L) {
            requiresZip64 = true;
        }
        if (!(zip64eocdlO = BinaryMapper.read(ZIP64_EOCDL, channel, eocdr.position - (long)ZipPrefixer.ZIP64_EOCDL.size)).isPresent() && requiresZip64) {
            throw new ZipException("This archive lacks a ZIP64 EOCDL, which is required according to its EOCDR.");
        }
        if (zip64eocdlO.isPresent()) {
            BinaryMapper.PatternInstance zip64eocdl = zip64eocdlO.get();
            LOG.fine(() -> String.format("ZIP64 EOCDL found at offset: \"0x%08X\"", zip64eocdl.position));
            cdOffset = zip64eocdl.getLong("relativeOffsetOfZip64EOCDR");
            if (mustAdjust) {
                writeQueue.add(zip64eocdl.writeLong("relativeOffsetOfZip64EOCDR", cdOffset += adjustment));
            }
            BinaryMapper.PatternInstance zip64eocdr = BinaryMapper.read(ZIP64_EOCDR, channel, cdOffset).orElseThrow(() -> new ZipException("Unable to find the ZIP64 EOCDR in the location given by ZIP64 EOCDL."));
            LOG.fine(() -> String.format("ZIP64 EOCDR found at offset: \"0x%08X\"", zip64eocdr.position));
            cdOffset = zip64eocdr.getLong("offsetOfStartOfCD");
            if (mustAdjust) {
                writeQueue.add(zip64eocdr.writeLong("offsetOfStartOfCD", cdOffset += adjustment));
            }
            numberOfCdEntries = zip64eocdr.getLong("numberOfEntriesInCDonThisDisk");
        }
        long sequentialOffset = cdOffset;
        ByteBuffer cfhBuffer = CFH.bufferFor();
        ByteBuffer lfhBuffer = LFH.bufferFor();
        int i = 0;
        while ((long)i < numberOfCdEntries) {
            BinaryMapper.PatternInstance cfh = BinaryMapper.read(CFH, channel, sequentialOffset, cfhBuffer).orElseThrow(() -> new ZipException("Central file header for entry is not where it should be"));
            LOG.fine(() -> String.format("CFH entry found at offset: \"0x%08X\"", cfh.position));
            sequentialOffset += (long)(cfh.spec.size + cfh.getUnsignedShort("fileNameLength"));
            int extraFieldLength = cfh.getUnsignedShort("extraFieldLength");
            long lfhOffset = cfh.getUnsignedInt("relativeOffsetOfLocalHeader");
            if (lfhOffset != 0xFFFFFFFFL) {
                if (mustAdjust) {
                    writeQueue.add(cfh.writeInt("relativeOffsetOfLocalHeader", (int)(lfhOffset += adjustment)));
                }
            } else {
                ArrayList<BinaryMapper.FieldSpec> requiredFieldsInEIEF = new ArrayList<BinaryMapper.FieldSpec>(5);
                requiredFieldsInEIEF.add(ZIP64_EIEF_SIGNATURE);
                requiredFieldsInEIEF.add(BinaryMapper.FieldSpec.of(2, "size"));
                if (cfh.getUnsignedInt("uncompressedSize") == 0xFFFFFFFFL) {
                    requiredFieldsInEIEF.add(BinaryMapper.FieldSpec.of(8, "uncompressedSize"));
                }
                if (cfh.getUnsignedInt("compressedSize") == 0xFFFFFFFFL) {
                    requiredFieldsInEIEF.add(BinaryMapper.FieldSpec.of(8, "compressedSize"));
                }
                requiredFieldsInEIEF.add(BinaryMapper.FieldSpec.of(8, "relativeOffsetOfLocalHeader"));
                BinaryMapper.PatternSpec requiredZip64EIEF = new BinaryMapper.PatternSpec(ByteOrder.LITTLE_ENDIAN, requiredFieldsInEIEF.toArray(new BinaryMapper.FieldSpec[0]));
                BinaryMapper.PatternInstance zip64eief = BinaryMapper.seek(requiredZip64EIEF, channel, sequentialOffset, patternInstance -> (long)patternInstance.getUnsignedShort("size") + 4L, sequentialOffset, sequentialOffset + (long)extraFieldLength).orElseThrow(() -> new ZipException("missing ZIP64 extra fields in CFH"));
                if (zip64eief.getUnsignedShort("size") < (requiredZip64EIEF.nameToFSI.size() - 2) * 8) {
                    throw new ZipException("ZIP64 extra fields in CFH seem to exist, but are too small.");
                }
                lfhOffset = zip64eief.getLong("relativeOffsetOfLocalHeader");
                if (mustAdjust) {
                    writeQueue.add(zip64eief.writeLong("relativeOffsetOfLocalHeader", lfhOffset += adjustment));
                }
            }
            BinaryMapper.PatternInstance lfh = BinaryMapper.read(LFH, channel, lfhOffset, lfhBuffer).orElseThrow(() -> new ZipException("Local file header for entry is not where it should be"));
            LOG.fine(() -> String.format("LFH entry found at offset: \"0x%08X\"", lfh.position));
            sequentialOffset += (long)(extraFieldLength + cfh.getUnsignedShort("fileCommentLength"));
            ++i;
        }
        return writeQueue;
    }

    static Path looksLikeZip(Path f) throws IOException {
        try (SeekableByteChannel channel = Files.newByteChannel(f, new OpenOption[0]);){
            BinaryMapper.seek(EOCDR, channel, Long.MAX_VALUE, false).orElseThrow(() -> new ZipException("Unable to locate EOCDR. This is probably not a ZIP file, or a broken one."));
        }
        return f;
    }

    static Path isUsableFile(Path f) throws IOException {
        if (!Files.isRegularFile(f, new LinkOption[0]) && !Files.isReadable(f)) {
            throw new IOException("path " + f + " is not a regular, readable file");
        }
        return f;
    }

    public static void main(String ... args) throws IOException {
        long endTime;
        if (args.length < 1) {
            System.out.println("usage: zip-prefixer zipfile [prefixfile ...]");
            System.exit(1);
        }
        Path zipfile = Paths.get(args[0], new String[0]);
        ZipPrefixer.looksLikeZip(ZipPrefixer.isUsableFile(zipfile));
        List<Path> paths = Arrays.stream(args, 1, args.length).map(x$0 -> Paths.get(x$0, new String[0])).collect(Collectors.toList());
        long startTime = System.nanoTime();
        if (paths.isEmpty()) {
            ZipPrefixer.validateZipOffsets(zipfile);
            endTime = System.nanoTime();
            System.out.print("validated offsets in " + zipfile);
        } else {
            long prefixesLength = ZipPrefixer.applyPrefixes(zipfile, paths);
            ZipPrefixer.adjustZipOffsets(zipfile, prefixesLength);
            endTime = System.nanoTime();
            System.out.printf("prefixed %d bytes on %s", prefixesLength, zipfile);
        }
        System.out.printf(" in %.1f ms %n", (double)(endTime - startTime) / 1000000.0);
    }
}

