/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2018 JFinal

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.http.common;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.ChannelContext;
import org.tio.core.Node;
import org.tio.core.exception.TioDecodeException;
import org.tio.http.common.HttpConst.RequestBodyFormat;
import org.tio.http.common.utils.HttpParseUtils;
import org.tio.http.common.utils.IpUtils;
import org.tio.utils.SysConst;
import org.tio.utils.hutool.StrUtil;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * http server中使用
 *
 * @author tanyaowu
 */
public class HttpRequestDecoder {
	private static final Logger log = LoggerFactory.getLogger(HttpRequestDecoder.class);

	public HttpRequestDecoder() {

	}

	/**
	 * @param buffer
	 * @param limit
	 * @param position
	 * @param readableLength
	 * @param channelContext
	 * @param httpConfig
	 * @return HttpRequest
	 * @throws TioDecodeException
	 */
	public static HttpRequest decode(ByteBuffer buffer, int limit, int position, int readableLength, ChannelContext channelContext, HttpConfig httpConfig)
		throws TioDecodeException {
		Map<String, String> headers = new HashMap<>();
		int contentLength;
		byte[] bodyBytes;
		RequestLine firstLine = parseRequestLine(buffer, httpConfig);
		if (firstLine == null) {
			return null;
		}
		// request header start
		boolean headerCompleted = parseHeaderLine(buffer, headers, 0, httpConfig);
		if (!headerCompleted) {
			return null;
		}
		String contentLengthStr = headers.get(HttpConst.RequestHeaderKey.Content_Length);

		if (StrUtil.isBlank(contentLengthStr)) {
			contentLength = 0;
		} else {
			contentLength = Integer.parseInt(contentLengthStr);
			if (contentLength > httpConfig.getMaxLengthOfPostBody()) {
				throw new TioDecodeException("post body length is too big[" + contentLength + "], max length is " + httpConfig.getMaxLengthOfPostBody() + " byte");
			}
		}

		int headerLength = (buffer.position() - position);
		int allNeedLength = headerLength + contentLength; //这个packet所需要的字节长度(含头部和体部)

		int notReceivedLength = allNeedLength - readableLength; //尚未接收到的数据长度
		if (notReceivedLength > 0) {
			if (notReceivedLength > channelContext.getReadBufferSize()) {
				channelContext.setReadBufferSize(notReceivedLength);
			}

			channelContext.setPacketNeededLength(allNeedLength);
			return null;
		}
		// request header end

		// ----------------------------------------------- request body start
		String realIp = IpUtils.getRealIp(channelContext, httpConfig, headers);
		if (httpConfig.checkHost && !headers.containsKey(HttpConst.RequestHeaderKey.Host)) {
			throw new TioDecodeException("there is no host header");
		}

		Node realNode;
		if (Objects.equals(realIp, channelContext.getClientNode().getIp())) {
			realNode = channelContext.getClientNode();
		} else {
			// realNode
			realNode = new Node(realIp, channelContext.getClientNode().getPort());
			channelContext.setProxyClientNode(realNode);
		}

		HttpRequest httpRequest = new HttpRequest(realNode);
		httpRequest.setRequestLine(firstLine);
		httpRequest.setChannelContext(channelContext);
		httpRequest.setHttpConfig(httpConfig);
		httpRequest.setHeaders(headers);
		httpRequest.setContentLength(contentLength);

		String connection = headers.get(HttpConst.RequestHeaderKey.Connection);
		if (connection != null) {
			httpRequest.setConnection(connection.toLowerCase());
		}

		if (StrUtil.isNotBlank(firstLine.queryString)) {
			decodeParams(httpRequest.getParams(), firstLine.queryString, httpRequest.getCharset());
		}

		if (contentLength > 0) {
			bodyBytes = new byte[contentLength];
			buffer.get(bodyBytes);
			httpRequest.setBody(bodyBytes);
			//解析消息体
			parseBody(httpRequest, firstLine, bodyBytes, channelContext, httpConfig);
		}
		// ----------------------------------------------- request body end
		return httpRequest;
	}

	/**
	 * @param params
	 * @param queryString
	 * @param charset
	 * @throws TioDecodeException
	 */
	public static void decodeParams(Map<String, Object[]> params, String queryString, Charset charset) throws TioDecodeException {
		if (StrUtil.isBlank(queryString)) {
			return;
		}

		String[] keyValues = queryString.split(SysConst.STR_AMP);
		for (String keyValue : keyValues) {
			String[] keyValueArr = keyValue.split(SysConst.STR_EQ);
			String value1 = null;
			if (keyValueArr.length == 2) {
				value1 = keyValueArr[1];
			} else if (keyValueArr.length > 2) {
				throw new TioDecodeException("含有多个" + SysConst.STR_EQ);
			}

			String key = keyValueArr[0];
			String value;
			if (StrUtil.isBlank(value1)) {
				value = null;
			} else {
				try {
					value = URLDecoder.decode(value1, charset.name());
				} catch (UnsupportedEncodingException e) {
					throw new TioDecodeException(e);
				}
			}
			Object[] existValue = params.get(key);
			String[] newExistValue;
			if (existValue == null) {
				newExistValue = new String[]{value};
			} else {
				newExistValue = new String[existValue.length + 1];
				System.arraycopy(existValue, 0, newExistValue, 0, existValue.length);
				newExistValue[newExistValue.length - 1] = value;
			}
			params.put(key, newExistValue);
		}
	}

	/**
	 * 解析消息体
	 *
	 * @param httpRequest
	 * @param firstLine
	 * @param bodyBytes
	 * @param channelContext
	 * @param httpConfig
	 * @throws TioDecodeException
	 */
	private static void parseBody(HttpRequest httpRequest, RequestLine firstLine, byte[] bodyBytes, ChannelContext channelContext, HttpConfig httpConfig)
		throws TioDecodeException {
		parseBodyFormat(httpRequest, httpRequest.getHeaders());
		RequestBodyFormat bodyFormat = httpRequest.getBodyFormat();

		httpRequest.setBody(bodyBytes);

		switch (bodyFormat) {
			case MULTIPART:
				if (log.isDebugEnabled()) {
					String bodyString;
					if (bodyBytes != null && bodyBytes.length > 0) {
						bodyString = new String(bodyBytes, httpRequest.getCharset());
						log.debug("{} multipart body value\r\n{}", channelContext, bodyString);
					}
				}

				//【multipart/form-data; boundary=----WebKitFormBoundaryuwYcfA2AIgxqIxA0】
				String contentType = httpRequest.getHeader(HttpConst.RequestHeaderKey.Content_Type);
				String initboundary = HttpParseUtils.getSubAttribute(contentType, "boundary");
				if (log.isDebugEnabled()) {
					log.debug("{}, initboundary:{}", channelContext, initboundary);
				}
				HttpMultiBodyDecoder.decode(httpRequest, firstLine, bodyBytes, initboundary, channelContext, httpConfig);
				break;
			default:
				String bodyString = null;
				if (bodyBytes != null && bodyBytes.length > 0) {
					bodyString = new String(bodyBytes, httpRequest.getCharset());
					httpRequest.setBodyString(bodyString);
					if (log.isInfoEnabled()) {
						log.info("{} body value\r\n{}", channelContext, bodyString);
					}
				}
				if (bodyFormat == RequestBodyFormat.URLENCODED) {
					parseUrlencoded(httpRequest, bodyString);
				}
				break;
		}
	}

	/**
	 * Content-Type : application/x-www-form-urlencoded; charset=UTF-8
	 * Content-Type : application/x-www-form-urlencoded; charset=UTF-8
	 *
	 * @param httpRequest HttpRequest
	 * @param headers     headers
	 */
	public static void parseBodyFormat(HttpRequest httpRequest, Map<String, String> headers) {
		String contentTypeStr = headers.get(HttpConst.RequestHeaderKey.Content_Type);
		String contentType = null;
		if (contentTypeStr != null) {
			contentType = contentTypeStr.toLowerCase();
		}
		if (contentType == null) {
			httpRequest.setBodyFormat(RequestBodyFormat.URLENCODED);
		} else if (contentType.startsWith(HttpConst.RequestHeaderValue.Content_Type.text_plain)) {
			httpRequest.setBodyFormat(RequestBodyFormat.TEXT);
		} else if (contentType.startsWith(HttpConst.RequestHeaderValue.Content_Type.multipart_form_data)) {
			httpRequest.setBodyFormat(RequestBodyFormat.MULTIPART);
		} else {
			httpRequest.setBodyFormat(RequestBodyFormat.URLENCODED);
		}
		if (StrUtil.isNotBlank(contentType)) {
			String charset = HttpParseUtils.getSubAttribute(contentType, "charset");
			if (StrUtil.isNotBlank(charset)) {
				httpRequest.setCharset(Charset.forName(charset));
			} else {
				httpRequest.setCharset(SysConst.DEFAULT_CHARSET);
			}
		}
	}

	/**
	 * 解析请求头的每一行
	 *
	 * @param buffer
	 * @param headers
	 * @param hasReceivedHeaderLength
	 * @param httpConfig
	 * @return 头部是否解析完成，true: 解析完成, false: 没有解析完成
	 * @throws TioDecodeException
	 */
	public static boolean parseHeaderLine(ByteBuffer buffer, Map<String, String> headers, int hasReceivedHeaderLength, HttpConfig httpConfig) throws TioDecodeException {
		byte[] allBs = buffer.array();
		int initPosition = buffer.position();
		int lastPosition = initPosition;
		int remaining = buffer.remaining();
		if (remaining == 0) {
			return false;
		} else if (remaining > 1) {
			byte b1 = buffer.get();
			byte b2 = buffer.get();
			if (SysConst.CR == b1 && SysConst.LF == b2) {
				return true;
			} else if (SysConst.LF == b1) {
				return true;
			}
		} else {
			if (SysConst.LF == buffer.get()) {
				return true;
			}
		}

		String name = null;
		String value = null;
		boolean hasValue = false;

		boolean needIteration = false;
		while (buffer.hasRemaining()) {
			byte b = buffer.get();
			if (name == null) {
				if (b == SysConst.COL) {
					int len = buffer.position() - lastPosition - 1;
					name = new String(allBs, lastPosition, len);
					lastPosition = buffer.position();
				} else if (b == SysConst.LF) {
					byte lastByte = buffer.get(buffer.position() - 2);
					int len = buffer.position() - lastPosition - 1;
					if (lastByte == SysConst.CR) {
						len = buffer.position() - lastPosition - 2;
					}
					name = new String(allBs, lastPosition, len);
					headers.put(name.toLowerCase(), "");
					needIteration = true;
					break;
				}
			} else if (value == null) {
				if (b == SysConst.LF) {
					byte lastByte = buffer.get(buffer.position() - 2);
					int len = buffer.position() - lastPosition - 1;
					if (lastByte == SysConst.CR) {
						len = buffer.position() - lastPosition - 2;
					}
					value = new String(allBs, lastPosition, len);

					headers.put(name.toLowerCase(), StrUtil.trimEnd(value));
					needIteration = true;
					break;
				} else {
					if (!hasValue && b == SysConst.SPACE) {
						lastPosition = buffer.position();
					} else {
						hasValue = true;
					}
				}
			}
		}

		// 这一行(header line)的字节数
		int lineLength = buffer.position() - initPosition;
		int maxLengthOfRequestLine = httpConfig.getMaxLengthOfRequestLine();
		if (lineLength > maxLengthOfRequestLine) {
			throw new TioDecodeException(name + " header line is too long, max length of header line is " + maxLengthOfRequestLine);
		}

		if (needIteration) {
			// header占用的字节数
			int headerLength = lineLength + hasReceivedHeaderLength;
			int maxLengthOfHeader = httpConfig.getMaxLengthOfHeader();
			if (headerLength > maxLengthOfHeader) {
				throw new TioDecodeException("header is too long, max length of header is " + maxLengthOfHeader);
			}
			return parseHeaderLine(buffer, headers, headerLength, httpConfig);
		}

		return false;
	}

	/**
	 * parse request line(the first line)
	 *
	 * @param buffer     GET /tio?value=tanyaowu HTTP/1.1
	 * @param httpConfig HttpConfig
	 * @return RequestLine
	 */
	public static RequestLine parseRequestLine(ByteBuffer buffer, HttpConfig httpConfig) throws TioDecodeException {
		byte[] allBs = buffer.array();
		int initPosition = buffer.position();
		String methodStr = null;
		String pathStr = null;
		String queryStr = null;
		String protocol = null;
		String version = null;
		int lastPosition = initPosition;
		while (buffer.hasRemaining()) {
			byte b = buffer.get();
			if (methodStr == null) {
				if (b == SysConst.SPACE) {
					int len = buffer.position() - lastPosition - 1;
					methodStr = new String(allBs, lastPosition, len);
					lastPosition = buffer.position();
				}
			} else if (pathStr == null) {
				if (b == SysConst.SPACE || b == SysConst.ASTERISK) {
					int len = buffer.position() - lastPosition - 1;
					pathStr = new String(allBs, lastPosition, len);
					lastPosition = buffer.position();
					if (b == SysConst.SPACE) {
						queryStr = SysConst.BLANK;
					}
				}
			} else if (queryStr == null) {
				if (b == SysConst.SPACE) {
					int len = buffer.position() - lastPosition - 1;
					queryStr = new String(allBs, lastPosition, len);
					lastPosition = buffer.position();
				}
			} else if (protocol == null) {
				if (b == SysConst.BACKSLASH) {
					int len = buffer.position() - lastPosition - 1;
					protocol = new String(allBs, lastPosition, len);
					lastPosition = buffer.position();
				}
			} else if (version == null) {
				if (b == SysConst.LF) {
					byte lastByte = buffer.get(buffer.position() - 2);
					int len = buffer.position() - lastPosition - 1;
					if (lastByte == SysConst.CR) {
						len = buffer.position() - lastPosition - 2;
					}
					version = new String(allBs, lastPosition, len);
					RequestLine requestLine = new RequestLine();
					Method method = Method.from(methodStr);
					requestLine.setMethod(method);
					requestLine.setPath(pathStr);
					requestLine.setInitPath(pathStr);
					requestLine.setQueryString(queryStr);
					requestLine.setProtocol(protocol);
					requestLine.setVersion(version);
					return requestLine;
				}
			}
		}
		int maxLengthOfRequestLine = httpConfig.getMaxLengthOfRequestLine();
		if ((buffer.position() - initPosition) > maxLengthOfRequestLine) {
			throw new TioDecodeException("request line is too long, max length of RequestLine is " + maxLengthOfRequestLine);
		}
		return null;
	}

	/**
	 * 解析URLENCODED格式的消息体
	 * 形如： 【Content-Type : application/x-www-form-urlencoded; charset=UTF-8】
	 *
	 * @throws TioDecodeException
	 */
	private static void parseUrlencoded(HttpRequest httpRequest, String bodyString) throws TioDecodeException {
		decodeParams(httpRequest.getParams(), bodyString, httpRequest.getCharset());
	}

}
