/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2020 t-io

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.intf.Packet;
import org.tio.core.intf.Packet.Meta;
import org.tio.core.ssl.SslFacadeContext;
import org.tio.core.stat.ChannelStat;
import org.tio.core.task.DecodeRunnable;
import org.tio.core.task.HandlerRunnable;
import org.tio.core.task.SendRunnable;
import org.tio.utils.hutool.StrUtil;
import org.tio.utils.prop.MapPropSupport;

import java.io.IOException;
import java.nio.channels.AsynchronousSocketChannel;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * @author tanyaowu
 * 2017年10月19日 上午9:39:46
 */
public abstract class ChannelContext extends MapPropSupport {
	private static final Logger log = LoggerFactory.getLogger(ChannelContext.class);
	public static final String UNKNOWN_ADDRESS_IP = "$UNKNOWN";
	public static final AtomicInteger UNKNOWN_ADDRESS_PORT_SEQ = new AtomicInteger();
	public final ReentrantReadWriteLock closeLock = new ReentrantReadWriteLock();
	public final ChannelStat stat = new ChannelStat();
	public final CloseMeta closeMeta = new CloseMeta();
	/**
	 * 此值不设时，心跳时间取org.tio.core.TioConfig.heartbeatTimeout
	 * 当然这个值如果小于org.tio.core.TioConfig.heartbeatTimeout，定时检查的时间间隔还是以org.tio.core.TioConfig.heartbeatTimeout为准，只是在判断时用此值
	 */
	public Long heartbeatTimeout = null;
	/**
	 * 一个packet所需要的字节数（用于应用告诉框架，下一次解码所需要的字节长度，省去冗余解码带来的性能损耗）
	 */
	public Integer packetNeededLength = null;
	public TioConfig tioConfig = null;
	public DecodeRunnable decodeRunnable = null;
	public HandlerRunnable handlerRunnable = null;
	public SendRunnable sendRunnable = null;
	public WriteCompletionHandler writeCompletionHandler = null;
	public SslFacadeContext sslFacadeContext;
	/**
	 * 状态位，使用二进制标识位来判断状态
	 *
	 * <p>
	 * 0~2 位，配置状态 isVirtual(虚拟用于压测):0,isReconnect(重连):0logWhenDecodeError(解码出现异常时，是否打印异常日志):0
	 * 3~5 位，连接状态 isWaitingClose(等待关闭):0,isClosed(已关闭):1,isRemoved(已关闭):0
	 * 6~7 位，扩展状态 isAccepted(已接受,用于业务例如：mqtt):0,isBizStatus(业务自定义状态):0
	 * </p>
	 */
	private byte states = 0;
	/**
	 * The asynchronous socket channel.
	 */
	public AsynchronousSocketChannel asynchronousSocketChannel;
	private ReadCompletionHandler readCompletionHandler = null;
	private String userId;
	private String token;
	private String bsId;
	private String id;
	private Node clientNode;
	/**
	 * 一些连接是代理的，譬如web服务器放在nginx后面，此时需要知道最原始的ip
	 */
	private Node proxyClientNode = null;
	private Node serverNode;
	/**
	 * 该连接在哪些组中
	 */
	private Set<String> groups = null;
	/**
	 * 个性化readBufferSize
	 */
	private Integer readBufferSize = null;
	/**
	 * 连接关闭的原因码
	 */
	private CloseCode closeCode = CloseCode.INIT_STATUS;

	/**
	 * ChannelContext
	 *
	 * @param tioConfig                 TioConfig
	 * @param asynchronousSocketChannel AsynchronousSocketChannel
	 */
	public ChannelContext(TioConfig tioConfig, AsynchronousSocketChannel asynchronousSocketChannel) {
		super();
		init(tioConfig, asynchronousSocketChannel);
		setUpSSL();
	}

	/**
	 * 创建一个虚拟ChannelContext，主要用来模拟一些操作，譬如压力测试，真实场景中用得少
	 *
	 * @param tioConfig TioConfig
	 */
	public ChannelContext(TioConfig tioConfig) {
		this(tioConfig, tioConfig.getTioUuid().uuid());
	}

	/**
	 * 创建一个虚拟ChannelContext，主要用来模拟一些操作，譬如压力测试，真实场景中用得少
	 *
	 * @param tioConfig TioConfig
	 * @param id        ChannelContext id
	 */
	public ChannelContext(TioConfig tioConfig, String id) {
		this.setVirtual(true);
		this.tioConfig = tioConfig;
		this.clientNode = new Node("127.0.0.1", 26254);
		this.id = id;
		if (StrUtil.isBlank(id)) {
			this.id = tioConfig.getTioUuid().uuid();
		}
		initOther();
	}

	protected void assignAnUnknownClientNode() {
		setClientNode(createUnknownNode());
	}

	public static Node createUnknownNode() {
		return new Node(UNKNOWN_ADDRESS_IP, UNKNOWN_ADDRESS_PORT_SEQ.incrementAndGet());
	}

	/**
	 * 设置 ssl
	 */
	public void setUpSSL() {
		if (tioConfig.sslConfig != null) {
			try {
				this.sslFacadeContext = new SslFacadeContext(this);
				if (tioConfig.isServer()) {
					this.sslFacadeContext.beginHandshake();
				}
			} catch (Exception e) {
				log.error("在开始SSL握手时发生了异常", e);
				Tio.close(this, "在开始SSL握手时发生了异常" + e.getMessage(), CloseCode.SSL_ERROR_ON_HANDSHAKE);
			}
		}
	}

	/**
	 * 创建Node
	 *
	 * @param asynchronousSocketChannel AsynchronousSocketChannel
	 * @return Node
	 * @throws IOException IOException
	 */
	public abstract Node createClientNode(AsynchronousSocketChannel asynchronousSocketChannel) throws IOException;

	/**
	 * @return the remoteNode
	 */
	public Node getClientNode() {
		return clientNode;
	}

	/**
	 * @param clientNode the clientNode to set
	 */
	public void setClientNode(Node clientNode) {
		if (!this.tioConfig.isShortConnection && this.clientNode != null) {
			tioConfig.clientNodes.remove(this);
		}

		this.clientNode = clientNode;
		if (this.tioConfig.isShortConnection) {
			return;
		}

		if (this.clientNode != null && !Objects.equals(UNKNOWN_ADDRESS_IP, this.clientNode.getIp())) {
			tioConfig.clientNodes.put(this);
		}
	}

	public Set<String> getGroups() {
		return groups;
	}

	/**
	 * @return the id
	 */
	public String getId() {
		return id;
	}

	/**
	 * @return the readCompletionHandler
	 */
	public ReadCompletionHandler getReadCompletionHandler() {
		return readCompletionHandler;
	}

	/**
	 * @return the serverNode
	 */
	public Node getServerNode() {
		return serverNode;
	}

	/**
	 * @param serverNode the serverNode to set
	 */
	public void setServerNode(Node serverNode) {
		this.serverNode = serverNode;
	}

	public String getToken() {
		return token;
	}

	public void setToken(String token) {
		this.token = token;
	}

	/**
	 * @return the writeCompletionHandler
	 */
	public WriteCompletionHandler getWriteCompletionHandler() {
		return writeCompletionHandler;
	}

	public void init(TioConfig tioConfig, AsynchronousSocketChannel asynchronousSocketChannel) {
		this.id = tioConfig.getTioUuid().uuid();
		this.setTioConfig(tioConfig);
		tioConfig.ids.bind(this);
		this.setAsynchronousSocketChannel(asynchronousSocketChannel);
		this.readCompletionHandler = new ReadCompletionHandler(this);
		this.writeCompletionHandler = new WriteCompletionHandler(this);
		this.setLogWhenDecodeError(tioConfig.logWhenDecodeError);
		initOther();
	}

	void initOther() {
		// 默认 closed 设置为 true
		setClosedState(true);
		if (!tioConfig.isShortConnection) {
			// 在长连接中，绑定群组几乎是必须要干的事，所以直接在初始化时给它赋值，省得在后面做同步处理
			groups = ConcurrentHashMap.newKeySet();
		}
	}

	/**
	 * @param packet        Packet
	 * @param isSentSuccess isSentSuccess
	 */
	public void processAfterSent(Packet packet, boolean isSentSuccess) {
		Meta meta = packet.getMeta();
		if (meta != null) {
			CountDownLatch countDownLatch = meta.getCountDownLatch();
			countDownLatch.countDown();
		}
		try {
			if (log.isDebugEnabled()) {
				log.debug("{} 已经发送 {}", this, packet.logstr());
			}
			//非SSL or SSL已经握手
			if (this.sslFacadeContext == null || this.sslFacadeContext.isHandshakeCompleted()) {
				if (tioConfig.getTioListener() != null) {
					try {
						tioConfig.getTioListener().onAfterSent(this, packet, isSentSuccess);
					} catch (Exception e) {
						log.error(e.getMessage(), e);
					}
				}
				if (tioConfig.statOn) {
					tioConfig.groupStat.sentPackets.increment();
					stat.sentPackets.incrementAndGet();
				}
			}
		} catch (Throwable e) {
			log.error(e.getMessage(), e);
		}
		if (packet.getPacketListener() != null) {
			try {
				packet.getPacketListener().onAfterSent(this, packet, isSentSuccess);
			} catch (Throwable e) {
				log.error(e.getMessage(), e);
			}
		}

	}

	/**
	 * @param asynchronousSocketChannel the asynchronousSocketChannel to set
	 */
	public void setAsynchronousSocketChannel(AsynchronousSocketChannel asynchronousSocketChannel) {
		this.asynchronousSocketChannel = asynchronousSocketChannel;
		if (asynchronousSocketChannel != null) {
			try {
				setClientNode(createClientNode(asynchronousSocketChannel));
			} catch (IOException e) {
				log.info(e.getMessage(), e);
				assignAnUnknownClientNode();
			}
		} else {
			assignAnUnknownClientNode();
		}
	}

	public boolean isVirtual() {
		return getState(0);
	}

	public void setVirtual(boolean virtual) {
		setState(0, virtual);
	}

	public boolean isReconnect() {
		return getState(1);
	}

	public void setReconnect(boolean isReconnect) {
		setState(1, isReconnect);
	}

	public boolean isLogWhenDecodeError() {
		return getState(2);
	}

	public void setLogWhenDecodeError(boolean logWhenDecodeError) {
		setState(2, logWhenDecodeError);
	}

	public boolean isWaitingClose() {
		return getState(3);
	}

	public void setWaitingClose(boolean waitingClose) {
		setState(3, waitingClose);
	}

	public boolean isClosed() {
		return getState(4);
	}

	private void setClosedState(boolean isClosed) {
		setState(4, isClosed);
	}

	/**
	 * @param isClosed the isClosed to set
	 */
	public void setClosed(boolean isClosed) {
		setClosedState(isClosed);
		if (isClosed && (clientNode == null || !UNKNOWN_ADDRESS_IP.equals(clientNode.getIp()))) {
			String before = this.toString();
			assignAnUnknownClientNode();
			log.info("关闭前{}, 关闭后{}", before, this);
		}
	}

	public boolean isRemoved() {
		return getState(5);
	}

	/**
	 * @param isRemoved the isRemoved to set
	 */
	public void setRemoved(boolean isRemoved) {
		setState(5, isRemoved);
	}

	public boolean isAccepted() {
		if (isClosed()) {
			return false;
		}
		return getState(6);
	}

	public void setAccepted(boolean accepted) {
		setState(6, accepted);
	}

	public boolean isBizStatus() {
		return getState(7);
	}

	public void setBizStatus(boolean bizStatus) {
		setState(7, bizStatus);
	}

	public void setPacketNeededLength(Integer packetNeededLength) {
		this.packetNeededLength = packetNeededLength;
	}

	public SslFacadeContext getSslFacadeContext() {
		return sslFacadeContext;
	}

	/**
	 * 设置 SslFacadeContext 用于动态 ssl 的过程
	 *
	 * @param sslFacadeContext sslFacadeContext
	 */
	public void setSslFacadeContext(SslFacadeContext sslFacadeContext) {
		this.sslFacadeContext = sslFacadeContext;
	}

	/**
	 * 获取 用户 id
	 *
	 * @return 用户 id
	 */
	public String getUserId() {
		return userId;
	}

	/**
	 * @param userId the userid to set
	 *               给框架内部用的，用户请勿调用此方法
	 */
	public void setUserId(String userId) {
		this.userId = userId;
	}

	/**
	 * @return the bsId
	 */
	public String getBsId() {
		return bsId;
	}

	/**
	 * @param bsId the bsId to set
	 */
	public void setBsId(String bsId) {
		this.bsId = bsId;
	}

	public TioConfig getTioConfig() {
		return tioConfig;
	}

	/**
	 * @param tioConfig the tioConfig to set
	 */
	private void setTioConfig(TioConfig tioConfig) {
		this.tioConfig = tioConfig;
		if (tioConfig != null) {
			decodeRunnable = new DecodeRunnable(this, tioConfig.tioExecutor);
			handlerRunnable = new HandlerRunnable(this, tioConfig.tioExecutor);
			sendRunnable = new SendRunnable(this, tioConfig.tioExecutor);
			tioConfig.connections.add(this);
		}
	}

	/**
	 * 是否是服务器端
	 *
	 * @return boolean
	 */
	public abstract boolean isServer();

	/**
	 * @return the heartbeatTimeout
	 */
	public Long getHeartbeatTimeout() {
		return heartbeatTimeout;
	}

	/**
	 * @param heartbeatTimeout the heartbeatTimeout to set
	 */
	public void setHeartbeatTimeout(Long heartbeatTimeout) {
		this.heartbeatTimeout = heartbeatTimeout;
	}

	public int getReadBufferSize() {
		if (readBufferSize != null && readBufferSize > 0) {
			return readBufferSize;
		}
		return this.tioConfig.getReadBufferSize();
	}

	public void setReadBufferSize(int readBufferSize) {
		this.readBufferSize = Math.min(readBufferSize, TcpConst.MAX_DATA_LENGTH);
	}

	/**
	 * @return the proxyClientNode
	 */
	public Node getProxyClientNode() {
		return proxyClientNode;
	}

	/**
	 * @param proxyClientNode the proxyClientNode to set
	 */
	public void setProxyClientNode(Node proxyClientNode) {
		this.proxyClientNode = proxyClientNode;
	}

	public CloseCode getCloseCode() {
		return closeCode;
	}

	public void setCloseCode(CloseCode closeCode) {
		this.closeCode = closeCode;
	}

	/**
	 * 获取指定状态位的值
	 *
	 * @param position 0~7 8位
	 */
	private boolean getState(int position) {
		return (this.states & (1 << position)) != 0;
	}

	/**
	 * 设置指定状态位的值
	 *
	 * @param position 0~7 8位
	 * @param state    状态
	 */
	private void setState(int position, boolean state) {
		if (state) {
			// 使用或运算将指定位设置为1
			this.states |= (byte) (1 << position);
		} else {
			// 使用与运算将指定位设置为0
			this.states &= (byte) ~(1 << position);
		}
	}

	/**
	 * 发送消息
	 *
	 * @param packet Packet
	 * @return 发送结果
	 */
	public boolean send(Packet packet) {
		return Tio.send(this, packet);
	}

	/**
	 * 发送消息
	 *
	 * @param packet Packet
	 * @return 发送结果
	 */
	public boolean bSend(Packet packet) {
		return Tio.bSend(this, packet);
	}

	/**
	 * 获取解码队列目前消息数
	 *
	 * @return 数据量
	 */
	public int getDecodeQueueSize() {
		return this.decodeRunnable.getMsgQueueSize();
	}

	/**
	 * 获取处理队列目前消息数
	 *
	 * @return 数据量
	 */
	public int getHandlerQueueSize() {
		return this.handlerRunnable.getMsgQueueSize();
	}

	/**
	 * 获取发送队列目前消息数
	 *
	 * @return 数据量
	 */
	public int getSendQueueSize() {
		return this.sendRunnable.getMsgQueueSize();
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		ChannelContext other = (ChannelContext) obj;
		if (id == null) {
			return other.id == null;
		} else {
			return id.equals(other.id);
		}
	}

	@Override
	public int hashCode() {
		if (StrUtil.isNotBlank(id)) {
			return this.id.hashCode();
		} else {
			return super.hashCode();
		}
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(64);
		if (serverNode != null) {
			sb.append("server:").append(serverNode);
		} else {
			sb.append("server:").append("NULL");
		}
		if (clientNode != null) {
			sb.append(", client:").append(clientNode);
		} else {
			sb.append(", client:").append("NULL");
		}
		if (this.isVirtual()) {
			sb.append(", virtual");
		}
		return sb.toString();
	}

	/**
	 * 连接关闭码
	 */
	public enum CloseCode {
		/**
		 * 没有提供原因码
		 */
		NO_CODE((byte) 1),
		/**
		 * 读异常
		 */
		READ_ERROR((byte) 2),
		/**
		 * 写异常
		 */
		WRITER_ERROR((byte) 3),
		/**
		 * 解码异常
		 */
		DECODE_ERROR((byte) 4),
		/**
		 * 通道未打开
		 */
		CHANNEL_NOT_OPEN((byte) 5),
		/**
		 * 读到的数据长度是0
		 */
		READ_COUNT_IS_ZERO((byte) 6),
		/**
		 * 对方关闭了连接
		 */
		CLOSED_BY_PEER((byte) 7),
		/**
		 * 读到的数据长度小于-1
		 */
		READ_COUNT_IS_NEGATIVE((byte) 8),
		/**
		 * 写数据长度小于0
		 */
		WRITE_COUNT_IS_NEGATIVE((byte) 9),
		/**
		 * 心跳超时
		 */
		HEARTBEAT_TIMEOUT((byte) 10),
		/**
		 * 连接失败
		 */
		CLIENT_CONNECTION_FAIL((byte) 80),

		/**
		 * SSL握手时发生异常
		 */
		SSL_ERROR_ON_HANDSHAKE((byte) 50),
		/**
		 * SSL session关闭了
		 */
		SSL_SESSION_CLOSED((byte) 51),
		/**
		 * SSL加密时发生异常
		 */
		SSL_ENCRYPTION_ERROR((byte) 52),
		/**
		 * SSL解密时发生异常
		 */
		SSL_DECRYPT_ERROR((byte) 53),

		/**
		 * 供用户使用
		 */
		USER_CODE_0((byte) 100),
		/**
		 * 供用户使用
		 */
		USER_CODE_1((byte) 101),
		/**
		 * 供用户使用
		 */
		USER_CODE_2((byte) 102),
		/**
		 * 供用户使用
		 */
		USER_CODE_3((byte) 103),
		/**
		 * 供用户使用
		 */
		USER_CODE_4((byte) 104),
		/**
		 * 供用户使用
		 */
		USER_CODE_5((byte) 105),
		/**
		 * 供用户使用
		 */
		USER_CODE_6((byte) 106),
		/**
		 * 供用户使用
		 */
		USER_CODE_7((byte) 107),
		/**
		 * 供用户使用
		 */
		USER_CODE_8((byte) 108),
		/**
		 * 供用户使用
		 */
		USER_CODE_9((byte) 109),
		/**
		 * 供用户使用
		 */
		USER_CODE_10((byte) 110),
		/**
		 * 初始值
		 */
		INIT_STATUS((byte) 199),
		/**
		 * 其它异常
		 */
		OTHER_ERROR((byte) 200),
		;

		final byte value;

		CloseCode(Byte value) {
			this.value = value;
		}

		public static CloseCode from(byte value) {
			CloseCode[] values = CloseCode.values();
			for (CloseCode v : values) {
				if (v.value == value) {
					return v;
				}
			}
			return null;
		}

		public byte getValue() {
			return value;
		}
	}

	public static class CloseMeta {
		public Throwable throwable;
		public String remark;
		public boolean isNeedRemove;

		public Throwable getThrowable() {
			return throwable;
		}

		public void setThrowable(Throwable throwable) {
			this.throwable = throwable;
		}

		public String getRemark() {
			return remark;
		}

		public void setRemark(String remark) {
			this.remark = remark;
		}

		public boolean isNeedRemove() {
			return isNeedRemove;
		}

		public void setNeedRemove(boolean isNeedRemove) {
			this.isNeedRemove = isNeedRemove;
		}
	}
}
