/*
	Apache License
	Version 2.0, January 2004
	http://www.apache.org/licenses/

	TERMS AND CONDITIONS FOR USE, REPRODUCTION, AND DISTRIBUTION

	1. Definitions.

	"License" shall mean the terms and conditions for use, reproduction, and
	distribution as defined by Sections 1 through 9 of this document.

	"Licensor" shall mean the copyright owner or entity authorized by the copyright
	owner that is granting the License.

	"Legal Entity" shall mean the union of the acting entity and all other entities
	that control, are controlled by, or are under common control with that entity.
	For the purposes of this definition, "control" means (i) the power, direct or
	indirect, to cause the direction or management of such entity, whether by
	contract or otherwise, or (ii) ownership of fifty percent (50%) or more of the
	outstanding shares, or (iii) beneficial ownership of such entity.

	"You" (or "Your") shall mean an individual or Legal Entity exercising
	permissions granted by this License.

	"Source" form shall mean the preferred form for making modifications, including
	but not limited to software source code, documentation source, and configuration
	files.

	"Object" form shall mean any form resulting from mechanical transformation or
	translation of a Source form, including but not limited to compiled object code,
	generated documentation, and conversions to other media types.

	"Work" shall mean the work of authorship, whether in Source or Object form, made
	available under the License, as indicated by a copyright notice that is included
	in or attached to the work (an example is provided in the Appendix below).

	"Derivative Works" shall mean any work, whether in Source or Object form, that
	is based on (or derived from) the Work and for which the editorial revisions,
	annotations, elaborations, or other modifications represent, as a whole, an
	original work of authorship. For the purposes of this License, Derivative Works
	shall not include works that remain separable from, or merely link (or bind by
	name) to the interfaces of, the Work and Derivative Works thereof.

	"Contribution" shall mean any work of authorship, including the original version
	of the Work and any modifications or additions to that Work or Derivative Works
	thereof, that is intentionally submitted to Licensor for inclusion in the Work
	by the copyright owner or by an individual or Legal Entity authorized to submit
	on behalf of the copyright owner. For the purposes of this definition,
	"submitted" means any form of electronic, verbal, or written communication sent
	to the Licensor or its representatives, including but not limited to
	communication on electronic mailing lists, source code control systems, and
	issue tracking systems that are managed by, or on behalf of, the Licensor for
	the purpose of discussing and improving the Work, but excluding communication
	that is conspicuously marked or otherwise designated in writing by the copyright
	owner as "Not a Contribution."

	"Contributor" shall mean Licensor and any individual or Legal Entity on behalf
	of whom a Contribution has been received by Licensor and subsequently
	incorporated within the Work.

	2. Grant of Copyright License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable copyright license to reproduce, prepare Derivative Works of,
	publicly display, publicly perform, sublicense, and distribute the Work and such
	Derivative Works in Source or Object form.

	3. Grant of Patent License.

	Subject to the terms and conditions of this License, each Contributor hereby
	grants to You a perpetual, worldwide, non-exclusive, no-charge, royalty-free,
	irrevocable (except as stated in this section) patent license to make, have
	made, use, offer to sell, sell, import, and otherwise transfer the Work, where
	such license applies only to those patent claims licensable by such Contributor
	that are necessarily infringed by their Contribution(s) alone or by combination
	of their Contribution(s) with the Work to which such Contribution(s) was
	submitted. If You institute patent litigation against any entity (including a
	cross-claim or counterclaim in a lawsuit) alleging that the Work or a
	Contribution incorporated within the Work constitutes direct or contributory
	patent infringement, then any patent licenses granted to You under this License
	for that Work shall terminate as of the date such litigation is filed.

	4. Redistribution.

	You may reproduce and distribute copies of the Work or Derivative Works thereof
	in any medium, with or without modifications, and in Source or Object form,
	provided that You meet the following conditions:

	You must give any other recipients of the Work or Derivative Works a copy of
	this License; and
	You must cause any modified files to carry prominent notices stating that You
	changed the files; and
	You must retain, in the Source form of any Derivative Works that You distribute,
	all copyright, patent, trademark, and attribution notices from the Source form
	of the Work, excluding those notices that do not pertain to any part of the
	Derivative Works; and
	If the Work includes a "NOTICE" text file as part of its distribution, then any
	Derivative Works that You distribute must include a readable copy of the
	attribution notices contained within such NOTICE file, excluding those notices
	that do not pertain to any part of the Derivative Works, in at least one of the
	following places: within a NOTICE text file distributed as part of the
	Derivative Works; within the Source form or documentation, if provided along
	with the Derivative Works; or, within a display generated by the Derivative
	Works, if and wherever such third-party notices normally appear. The contents of
	the NOTICE file are for informational purposes only and do not modify the
	License. You may add Your own attribution notices within Derivative Works that
	You distribute, alongside or as an addendum to the NOTICE text from the Work,
	provided that such additional attribution notices cannot be construed as
	modifying the License.
	You may add Your own copyright statement to Your modifications and may provide
	additional or different license terms and conditions for use, reproduction, or
	distribution of Your modifications, or for any such Derivative Works as a whole,
	provided Your use, reproduction, and distribution of the Work otherwise complies
	with the conditions stated in this License.

	5. Submission of Contributions.

	Unless You explicitly state otherwise, any Contribution intentionally submitted
	for inclusion in the Work by You to the Licensor shall be under the terms and
	conditions of this License, without any additional terms or conditions.
	Notwithstanding the above, nothing herein shall supersede or modify the terms of
	any separate license agreement you may have executed with Licensor regarding
	such Contributions.

	6. Trademarks.

	This License does not grant permission to use the trade names, trademarks,
	service marks, or product names of the Licensor, except as required for
	reasonable and customary use in describing the origin of the Work and
	reproducing the content of the NOTICE file.

	7. Disclaimer of Warranty.

	Unless required by applicable law or agreed to in writing, Licensor provides the
	Work (and each Contributor provides its Contributions) on an "AS IS" BASIS,
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied,
	including, without limitation, any warranties or conditions of TITLE,
	NON-INFRINGEMENT, MERCHANTABILITY, or FITNESS FOR A PARTICULAR PURPOSE. You are
	solely responsible for determining the appropriateness of using or
	redistributing the Work and assume any risks associated with Your exercise of
	permissions under this License.

	8. Limitation of Liability.

	In no event and under no legal theory, whether in tort (including negligence),
	contract, or otherwise, unless required by applicable law (such as deliberate
	and grossly negligent acts) or agreed to in writing, shall any Contributor be
	liable to You for damages, including any direct, indirect, special, incidental,
	or consequential damages of any character arising as a result of this License or
	out of the use or inability to use the Work (including but not limited to
	damages for loss of goodwill, work stoppage, computer failure or malfunction, or
	any and all other commercial damages or losses), even if such Contributor has
	been advised of the possibility of such damages.

	9. Accepting Warranty or Additional Liability.

	While redistributing the Work or Derivative Works thereof, You may choose to
	offer, and charge a fee for, acceptance of support, warranty, indemnity, or
	other liability obligations and/or rights consistent with this License. However,
	in accepting such obligations, You may act only on Your own behalf and on Your
	sole responsibility, not on behalf of any other Contributor, and only if You
	agree to indemnify, defend, and hold each Contributor harmless for any liability
	incurred by, or claims asserted against, such Contributor by reason of your
	accepting any such warranty or additional liability.

	END OF TERMS AND CONDITIONS

	APPENDIX: How to apply the Apache License to your work

	To apply the Apache License to your work, attach the following boilerplate
	notice, with the fields enclosed by brackets "{}" replaced with your own
	identifying information. (Don't include the brackets!) The text should be
	enclosed in the appropriate comment syntax for the file format. We also
	recommend that a file or class name and description of purpose be included on
	the same "printed page" as the copyright notice for easier identification within
	third-party archives.

	   Copyright 2020 t-io

	   Licensed under the Apache License, Version 2.0 (the "License");
	   you may not use this file except in compliance with the License.
	   You may obtain a copy of the License at

	     http://www.apache.org/licenses/LICENSE-2.0

	   Unless required by applicable law or agreed to in writing, software
	   distributed under the License is distributed on an "AS IS" BASIS,
	   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	   See the License for the specific language governing permissions and
	   limitations under the License.
*/
package org.tio.server;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tio.core.Node;
import org.tio.server.task.ServerHeartbeatTask;
import org.tio.utils.SysConst;
import org.tio.utils.Version;
import org.tio.utils.hutool.ClassUtil;
import org.tio.utils.hutool.DateUtil;
import org.tio.utils.hutool.StrUtil;
import org.tio.utils.timer.DefaultTimerTaskService;
import org.tio.utils.timer.TimerTask;
import org.tio.utils.timer.TimerTaskService;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.StandardSocketOptions;
import java.nio.channels.AsynchronousChannelGroup;
import java.nio.channels.AsynchronousServerSocketChannel;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

/**
 * @author tanyaowu
 */
public class TioServer {
	private static final Logger log = LoggerFactory.getLogger(TioServer.class);
	private final Node serverNode;
	private final TioServerConfig serverConfig;
	private AsynchronousServerSocketChannel serverSocketChannel;
	private AsynchronousChannelGroup channelGroup = null;
	private TimerTaskService taskService;
	private boolean isWaitingStop = false;

	public TioServer(int port, TioServerConfig serverConfig) {
		this(null, port, serverConfig);
	}

	public TioServer(String ip, int port, TioServerConfig serverConfig) {
		this(new Node(ip, port), serverConfig);
	}

	public TioServer(Node serverNode, TioServerConfig serverConfig) {
		this.serverNode = Objects.requireNonNull(serverNode, "serverNode 不能为 null");
		this.serverConfig = Objects.requireNonNull(serverConfig, "TioServerConfig 不能为 null");
	}

	/**
	 * @return the tioServerConfig
	 */
	public TioServerConfig getServerConfig() {
		return serverConfig;
	}

	/**
	 * @return the serverNode
	 */
	public Node getServerNode() {
		return serverNode;
	}

	/**
	 * @return the serverSocketChannel
	 */
	public AsynchronousServerSocketChannel getServerSocketChannel() {
		return serverSocketChannel;
	}

	/**
	 * @return the isWaitingStop
	 */
	public boolean isWaitingStop() {
		return isWaitingStop;
	}

	/**
	 * 定时任务：发心跳
	 */
	private void startHeartbeatTask() {
		// 先判断是否开启心跳检测
		if (serverConfig.heartbeatTimeout > 0) {
			this.taskService.addTask(systemTimer -> new ServerHeartbeatTask(systemTimer, serverConfig));
		} else if (this.serverConfig.isNeedCheckHeartbeat()) {
			log.warn("用户取消了 mica-net 的心跳定时发送功能，请确认是否自定义心跳机制");
		}
	}

	/**
	 * 添加定时任务
	 *
	 * @param command runnable
	 * @param delay   delay
	 * @return TimerTask
	 */
	public TimerTask schedule(Runnable command, long delay) {
		return schedule(command, delay, null);
	}

	/**
	 * 添加定时任务
	 *
	 * @param command  runnable
	 * @param delay    delay
	 * @param executor 用于自定义线程池，处理耗时业务
	 * @return TimerTask
	 */
	public TimerTask schedule(Runnable command, long delay, Executor executor) {
		return this.taskService.addTask((systemTimer -> new TimerTask(delay) {
			@Override
			public void run() {
				try {
					// 1. 再次添加 任务
					systemTimer.add(this);
					// 2. 执行任务
					if (executor == null) {
						command.run();
					} else {
						executor.execute(command);
					}
				} catch (Exception e) {
					log.error("mica-net server schedule error", e);
				}
			}
		}));
	}

	/**
	 * 添加定时任务，注意：如果抛出异常，会终止后续任务，请自行处理异常
	 *
	 * @param command runnable
	 * @param delay   delay
	 * @return TimerTask
	 */
	public TimerTask scheduleOnce(Runnable command, long delay) {
		return scheduleOnce(command, delay, null);
	}

	/**
	 * 添加定时任务，注意：如果抛出异常，会终止后续任务，请自行处理异常
	 *
	 * @param command  runnable
	 * @param delay    delay
	 * @param executor 用于自定义线程池，处理耗时业务
	 * @return TimerTask
	 */
	public TimerTask scheduleOnce(Runnable command, long delay, Executor executor) {
		return this.taskService.addTask((systemTimer -> new TimerTask(delay) {
			@Override
			public void run() {
				try {
					if (executor == null) {
						command.run();
					} else {
						executor.execute(command);
					}
				} catch (Exception e) {
					log.error("mica-net server schedule once error", e);
				}
			}
		}));
	}

	public void start() throws IOException {
		long start = System.currentTimeMillis();
		// 配置 time task
		this.taskService = getTimerTaskService(this.serverConfig);
		// 启动任务服务
		this.taskService.start();
		// 启动心跳检测任务
		startHeartbeatTask();
		channelGroup = AsynchronousChannelGroup.withThreadPool(serverConfig.groupExecutor);
		serverSocketChannel = AsynchronousServerSocketChannel.open(channelGroup);

		serverSocketChannel.setOption(StandardSocketOptions.SO_REUSEADDR, true);
		serverSocketChannel.setOption(StandardSocketOptions.SO_RCVBUF, 64 * 1024);

		String serverIp = serverNode.getIp();
		InetSocketAddress listenAddress;
		if (StrUtil.isBlank(serverIp)) {
			listenAddress = new InetSocketAddress(serverNode.getPort());
		} else {
			listenAddress = new InetSocketAddress(serverIp, serverNode.getPort());
		}

		serverSocketChannel.bind(listenAddress, serverConfig.getBacklog());

		AcceptCompletionHandler acceptCompletionHandler = serverConfig.getAcceptCompletionHandler();
		serverSocketChannel.accept(this, acceptCompletionHandler);

		serverConfig.startTime = System.currentTimeMillis();

		//下面这段代码有点无聊，写得随意，纯粹是为了打印好看些
		String baseStr = "|----------------------------------------------------------------------------------------|";
		int baseLen = baseStr.length();
		int xxLen = 18;
		int aaLen = baseLen - 3;
		StackTraceElement[] ses = Thread.currentThread().getStackTrace();
		StackTraceElement se = ses[ses.length - 1];
		List<String> infoList = new ArrayList<>();
		// 打印启动信息
		String serverConfigName = serverConfig.getName();
		if (StrUtil.isNotBlank(serverConfigName)) {
			infoList.add(StrUtil.fillAfter("TioConfig name", ' ', xxLen) + "| " + serverConfig.getName());
		}
		// 版本信息
		infoList.add(StrUtil.fillAfter("Mica net version", ' ', xxLen) + "| " + Version.getVersion());
		infoList.add(StrUtil.fillAfter("Started at", ' ', xxLen) + "| " + DateUtil.formatDateTime(LocalDateTime.now()));
		infoList.add(StrUtil.fillAfter("Listen on", ' ', xxLen) + "| " + this.serverNode);
		infoList.add(StrUtil.fillAfter("Main Class", ' ', xxLen) + "| " + se.getClassName());
		// 启动的时间和pid信息
		startManagementDebugInfo(infoList, xxLen, start);
		// 100
		StringBuilder printStr = new StringBuilder(SysConst.CRLF + baseStr + SysConst.CRLF);
		for (String string : infoList) {
			printStr.append("| ").append(StrUtil.fillAfter(string, ' ', aaLen)).append("|\r\n");
		}
		printStr.append(baseStr).append(SysConst.CRLF);
		if (log.isInfoEnabled()) {
			log.info(printStr.toString());
		} else {
			System.out.println(printStr);
		}
	}

	/**
	 * 获取 time task 服务
	 *
	 * @param serverConfig TioServerConfig
	 * @return TimerTaskService
	 */
	private static TimerTaskService getTimerTaskService(TioServerConfig serverConfig) {
		return Optional.ofNullable(serverConfig.getTaskService()).orElseGet(DefaultTimerTaskService::new);
	}

	/**
	 * 启动信息打印
	 *
	 * @param infoList infoList
	 * @param xxLen    xxLen
	 * @param start    start
	 */
	private static void startManagementDebugInfo(List<String> infoList, int xxLen, long start) {
		// Android 中没有 ManagementFactory
		boolean hasManagementFactory = ClassUtil.isPresent("java.lang.management.ManagementFactory");
		if (hasManagementFactory) {
			try {
				java.lang.management.RuntimeMXBean runtimeMxBean = java.lang.management.ManagementFactory.getRuntimeMXBean();
				long startTime = runtimeMxBean.getStartTime();
				long startCost = System.currentTimeMillis() - startTime;
				String runtimeName = runtimeMxBean.getName();
				String pid = runtimeName.split("@")[0];
				infoList.add(StrUtil.fillAfter("Jvm start time", ' ', xxLen) + "| " + startCost + "ms");
				infoList.add(StrUtil.fillAfter("Tio start time", ' ', xxLen) + "| " + (System.currentTimeMillis() - start) + "ms");
				infoList.add(StrUtil.fillAfter("Pid", ' ', xxLen) + "| " + pid);
			} catch (Throwable ignore) {
				// ignore
			}
		}
	}

	/**
	 * @return 是否停止成功
	 */
	public boolean stop() {
		isWaitingStop = true;
		// 删除实例
		serverConfig.remove();
		// 停止 taskService
		this.taskService.stop();
		try {
			channelGroup.shutdownNow();
		} catch (Exception e) {
			log.error("channelGroup.shutdownNow()时报错", e);
		}
		try {
			serverSocketChannel.close();
		} catch (Exception e1) {
			log.error("serverSocketChannel.close()时报错", e1);
		}
		// 停止心跳线程
		serverConfig.setStopped(true);
		try {
			serverConfig.groupExecutor.shutdown();
		} catch (Exception e1) {
			log.error(e1.getMessage(), e1);
		}
		try {
			serverConfig.tioExecutor.shutdown();
		} catch (Exception e1) {
			log.error(e1.getMessage(), e1);
		}
		boolean ret;
		try {
			ret = serverConfig.groupExecutor.awaitTermination(6000, TimeUnit.SECONDS);
			ret = ret && serverConfig.tioExecutor.awaitTermination(6000, TimeUnit.SECONDS);
		} catch (InterruptedException e) {
			ret = false;
			Thread.currentThread().interrupt();
			log.error(e.getMessage(), e);
		}
		log.info("{} stopped ret:{}", this.serverNode, ret);
		return ret;
	}

}
