package net.dona.doip.server;

import java.security.PublicKey;
import java.security.cert.X509Certificate;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import net.dona.doip.InDoipMessage;

/**
 * Interface for an incoming DOIP request to be handled by a {@link DoipProcessor}.
 */
public interface DoipServerRequest {

    /**
     * Returns the client ID given by the initial segment of this DOIP request.
     * 
     * @return the client ID given by the initial segment of this DOIP request
     */
    String getClientId();

    /**
     * Returns the target ID (the object on which to perform the operation) given by the initial segment of this DOIP request.
     * 
     * @return the target ID given by the initial segment of this DOIP request
     */
    String getTargetId();

    /**
     * Returns the operation ID (the operation to perform) given by the initial segment of this DOIP request.
     * 
     * @return the operation ID given by the initial segment of this DOIP request
     */
    String getOperationId();

    /**
     * Returns the attributes given by the initial segment of this DOIP request.
     * 
     * @return the attributes given by the initial segment of this DOIP request
     */
    JsonObject getAttributes();

    /**
     * Returns a single attribute from the attributes given by the initial segment of the DOIP request.
     *
     * @param key the attribute to retrieve
     * @return a single attribute from the attributes given by the initial segment of the DOIP request
     */
    JsonElement getAttribute(String key);

    /**
     * Returns as a String a single attribute from the attributes given by the initial segment of this DOIP request.
     *
     * @param key the attribute to retrieve
     * @return a single attribute as a string from the attributes given by the initial segment of this DOIP request
     */
    String getAttributeAsString(String key);

    /**
     * Returns the authentication information given by the initial segment of this DOIP request.
     * 
     * @return the authentication information given by the initial segment of this DOIP request
     */
    JsonElement getAuthentication();

    /**
     * Returns the input of the request.  In the case of a "compact" single-segment request, this will
     * be a single JSON segment corresponding to the "input" property of the single-segment request.
     * Otherwise it will be all remaining segments of the request after the initial segment.
     * 
     * @return the input of this DOIP request
     */
    InDoipMessage getInput();

    /**
     * If the client is using a TLS client-side certificate, this returns the client id from the certificate.
     * 
     * @return the client ID from a TLS client-side certificate
     */
    String getConnectionClientId();

    /**
     * If the client is using a TLS client-side certificate, this returns the public key from the certificate.
     * 
     * @return the public key from a TLS client-side certificate
     */
    PublicKey getConnectionPublicKey();

    /**
     * If the client is using a TLS client-side certificate, this returns the supplied certificate chain.
     * 
     * @return the supplied certificate chain from a a TLS client-side certificate
     */
    X509Certificate[] getConnectionCertificateChain();
}
