package net.dona.doip.client;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

/**
 * An exception thrown by {@link DoipClient} when something goes wrong such as an unexpected status code returned for an operation.
 */
public class DoipException extends Exception {

    private final String statusCode;
    private final JsonElement response;

    public DoipException(String message) {
        super(message);
        this.statusCode = null;
        response = responseForMessage(message);
    }

    public DoipException(JsonElement response) {
        super(messageForResponse(response));
        this.statusCode = null;
        this.response = response;
    }

    public DoipException(String statusCode, String message) {
        super(message);
        this.statusCode = statusCode;
        response = responseForMessage(message);
    }

    public DoipException(String statusCode, JsonElement response) {
        super(messageForResponse(response));
        this.statusCode = statusCode;
        this.response = response;
    }

    public DoipException(Throwable cause) {
        super(cause);
        this.statusCode = null;
        this.response = null;
    }

    public DoipException(String statusCode, String message, Throwable cause) {
        super(message, cause);
        this.statusCode = statusCode;
        response = responseForMessage(message);
    }

    public DoipException(String statusCode, JsonElement response, Throwable cause) {
        super(messageForResponse(response), cause);
        this.statusCode = statusCode;
        this.response = response;
    }

    /**
     * Gets the status code returned by the operation, if available.
     *
     * @return the status code returned by the operation, if available
     */
    public String getStatusCode() {
        return statusCode;
    }

    public JsonElement getResponse() {
        return response;
    }

    public static String messageForResponse(JsonElement response) {
        if (response == null || response.isJsonNull()) return null;
        try {
            if (response.isJsonObject()) {
                return response.getAsJsonObject().get("message").getAsString();
            } else if (response.isJsonPrimitive()) {
                return response.getAsString();
            } else {
                return null;
            }
        } catch (Exception e) {
            return null;
        }
    }

    public static JsonElement responseForMessage(String message) {
        if (message == null) return null;
        JsonObject res = new JsonObject();
        res.addProperty("message", message);
        return res;
    }
}
