package net.dona.doip.client;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;
import net.dona.doip.*;
import net.dona.doip.client.transport.*;
import net.dona.doip.util.GsonUtility;
import net.handle.hdllib.HandleException;
import net.handle.hdllib.HandleResolver;
import net.handle.hdllib.HandleValue;

import java.util.*;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A DOIP client for performing operations on objects.  The client can be used to perform arbitrary operations on object, and also provides
 * specific methods for the basic DOIP operations.
 *
 * In general handle resolution will be used to find the service information for accessing the object: the
 * target id is resolved, handle values of type DOIPService are references to service ids which are resolved,
 * handle values of type DOIPServiceInfo have service connection information.
 *
 * It is also possible to explicitly supply the service through which the operation is to be performed.
 *
 * The user should call {@link #close()} to release all resources.
 */
public class DoipClient extends AbstractDoipClient implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(DoipClient.class);

    private static final String DOIP_SERVICE_INFO = "DOIPServiceInfo";
    private static final String TYPE_DOIP_SERVICE_INFO = "0.TYPE/DOIPServiceInfo";
    private static final String DOIP_SERVICE = "DOIPService";
    private static final String TYPE_DOIP_SERVICE = "0.TYPE/DOIPService";

    // connections per service handle
    private static final int MAX_POOL_SIZE = 100;
    private static final int MAX_HOP_COUNT = 20;

    private final Cache<String, ServiceInfoAndPool> serviceHandleToPoolsMap;
    private final Cache<String, String> targetIdToServiceHandleMap;

    private final TransportDoipClient doipClient;
    private final HandleResolver resolver;

    private boolean closed; // guarded by synchronized methods

    /**
     * Constructs a new DoipClient.
     */
    public DoipClient() {
        doipClient = new TransportDoipClient();
        resolver = new HandleResolver();
        serviceHandleToPoolsMap = CacheBuilder.newBuilder()
                .expireAfterWrite(1, TimeUnit.HOURS)
                .removalListener(new PoolRemovalListener())
                .build();
        targetIdToServiceHandleMap = CacheBuilder.newBuilder()
                .expireAfterWrite(1, TimeUnit.HOURS)
                .build();
    }

    /**
     * Closes all open connections and release all resources.
     */
    @Override
    public synchronized void close() {
        closed = true;
        for (ServiceInfoAndPool serviceInfoAndPool : serviceHandleToPoolsMap.asMap().values()) {
            try {
                serviceInfoAndPool.pool.shutdown();
            } catch (Exception e) {
                logger.warn("Error closing", e);
            }
        }
        try {
            doipClient.close();
        } catch (Exception e) {
            logger.warn("Error closing", e);
        }
    }

    /**
     * Performs an operation at a specified service.
     *
     * @param headers the content of the initial segment of the request
     * @param input the input to the operation as an InDoipMessage
     * @param serviceInfo the service at which to perform the operation
     * @return the response
     * @throws DoipException if an operation with headers, a DOIP message to be read as input, and a connection and pool cannot be performed
     */
    @Override
    public DoipClientResponse performOperation(DoipRequestHeaders headers, InDoipMessage input, ServiceInfo serviceInfo) throws DoipException {
        ConnectionAndPool connectionAndPool = connectionAndPoolForOptions(serviceInfo, headers.targetId);
        return performOperationWithConnection(headers, input, connectionAndPool);
    }

    private ConnectionAndPool connectionAndPoolForOptions(ServiceInfo serviceInfo, String targetId) throws DoipException {
        ConnectionAndPool connectionAndPool;
        if (serviceInfo == null) {
            connectionAndPool = getConnectionFor(targetId);
        } else if (serviceInfo.ipAddress != null) {
            ServiceInfoAndPool serviceInfoAndPool = getOrCreatePool(serviceInfo);
            connectionAndPool = new ConnectionAndPool(serviceInfoAndPool.pool);
        } else if (serviceInfo.serviceId != null) {
            connectionAndPool = getConnectionFor(serviceInfo.serviceId);
        } else {
            throw new DoipException("Missing options");
        }
        return connectionAndPool;
    }

    @SuppressWarnings("resource")
    private DoipClientResponse performOperationWithConnection(DoipRequestHeaders headers, InDoipMessage input, ConnectionAndPool connectionAndPool) throws DoipException {
        DoipConnection conn = connectionAndPool.getConnection();
        DoipClientResponse response;
        try {
            if (input != null) {
                response = conn.sendRequest(headers, input);
            } else {
                response = conn.sendCompactRequest(headers);
            }
        } catch (Exception e) {
            try {
                // This connection is potentially no longer usable
                try {
                    conn.close();
                } catch (Exception ex) {
                    e.addSuppressed(ex);
                }
                connectionAndPool.releaseConnection();
            } catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
            }
            throw new DoipException(e);
        }
        response.setOnClose(() -> {
            try {
                connectionAndPool.releaseConnection();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        });
        return response;
    }

    private ConnectionAndPool getConnectionFor(String targetId) throws DoipException {
        String serviceHandle = targetIdToServiceHandleMap.getIfPresent(targetId);
        ServiceInfoAndPool serviceInfoAndPool;
        if (serviceHandle == null) {
            serviceInfoAndPool = getServiceInfoAndPoolFor(targetId);
            targetIdToServiceHandleMap.put(targetId, serviceInfoAndPool.serviceInfo.serviceId);
        } else {
            serviceInfoAndPool = serviceHandleToPoolsMap.getIfPresent(serviceHandle);
            if (serviceInfoAndPool == null) {
                serviceInfoAndPool = getServiceInfoAndPoolFor(serviceHandle);
            }
        }
        ConnectionAndPool result = new ConnectionAndPool(serviceInfoAndPool.pool);
        return result;
    }

    private ServiceInfoAndPool getServiceInfoAndPoolFor(String handle) throws DoipException {
        try {
            ServiceInfo serviceInfo = getServiceInfoFor(handle, 0);
            if (serviceInfo == null) {
                throw new DoipException("DOIPServiceInfo not found for " + handle);
            }
            ServiceInfoAndPool serviceInfoAndPool = getOrCreatePool(serviceInfo);
            return serviceInfoAndPool;
        } catch (HandleException he) {
            throw new DoipException(he);
        }
    }

    private synchronized ServiceInfoAndPool getOrCreatePool(ServiceInfo serviceInfo) {
        if (closed) throw new IllegalStateException("closed");
        ServiceInfoAndPool serviceInfoAndPool = serviceHandleToPoolsMap.getIfPresent(serviceInfo.serviceId);
        if (serviceInfoAndPool == null) {
            DoipConnectionPool pool = new DoipConnectionPool(MAX_POOL_SIZE, doipClient, connectionOptionsForServiceInfo(serviceInfo));
            serviceInfoAndPool = new ServiceInfoAndPool(serviceInfo, pool);
            serviceHandleToPoolsMap.put(serviceInfo.serviceId, serviceInfoAndPool);
        }
        return serviceInfoAndPool;
    }

    private ConnectionOptions connectionOptionsForServiceInfo(ServiceInfo serviceInfo) {
        ConnectionOptions res = new ConnectionOptions();
        res.serverId = serviceInfo.serviceId;
        res.address = serviceInfo.ipAddress;
        res.port = serviceInfo.port;
        if (serviceInfo.publicKey != null) {
            res.trustedServerPublicKeys = Collections.singletonList(serviceInfo.publicKey);
        }
        return res;
    }

    private static class ServiceInfoAndPool {
        public final ServiceInfo serviceInfo;
        public final DoipConnectionPool pool;

        public ServiceInfoAndPool(ServiceInfo serviceInfo, DoipConnectionPool pool) {
            this.serviceInfo = serviceInfo;
            this.pool = pool;
        }
    }

    private static class PoolRemovalListener implements RemovalListener<String, ServiceInfoAndPool> {

        @Override
        public void onRemoval(RemovalNotification<String, ServiceInfoAndPool> notification) {
            ServiceInfoAndPool serviceInfoAndPool = notification.getValue();
            serviceInfoAndPool.pool.shutdown();
        }
    }

    private ServiceInfo getServiceInfoFor(String handle, int hopCount) throws HandleException {
        HandleValue[] values = resolver.resolveHandle(handle, new String[] { DOIP_SERVICE, TYPE_DOIP_SERVICE, DOIP_SERVICE_INFO, TYPE_DOIP_SERVICE_INFO }, null);
        for (HandleValue value : values) {
            String type = value.getTypeAsString();
            if (DOIP_SERVICE_INFO.equals(type) || TYPE_DOIP_SERVICE_INFO.equals(type)) {
                String json = value.getDataAsString();
                DigitalObject dobj = GsonUtility.getGson().fromJson(json, DigitalObject.class);
                ServiceInfo result = GsonUtility.getGson().fromJson(dobj.attributes, ServiceInfo.class);
                result.serviceId = handle;
                return result;
            }
        }
        for (HandleValue value : values) {
            String type = value.getTypeAsString();
            if (DOIP_SERVICE.equals(type) || TYPE_DOIP_SERVICE.equals(type)) {
                String doipServiceHandle = value.getDataAsString();
                if (hopCount >= MAX_HOP_COUNT) {
                    return null;
                }
                return getServiceInfoFor(doipServiceHandle, hopCount+1);
            }
        }
        return null;
    }
}
