package net.dona.doip.client;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.reflect.TypeToken;

import net.dona.doip.*;
import net.dona.doip.client.transport.DoipClientResponse;
import net.dona.doip.util.ErrorMessageUtil;
import net.dona.doip.util.GsonUtility;
import net.dona.doip.util.InDoipMessageUtil;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;

public abstract class AbstractDoipClient implements DoipClientInterface {

    @Override
    public void close() {

    }

    @Override
    public JsonElement performOperationWithJsonResponse(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes) throws DoipException {
        return performOperationWithJsonResponse(targetId, operationId, authInfo, attributes, null);
    }

    @Override
    public JsonElement performOperationWithJsonResponse(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, ServiceInfo serviceInfo) throws DoipException {
        return performOperationWithJsonResponse(targetId, operationId, authInfo, attributes, null, serviceInfo);
    }

    @Override
    public JsonElement performOperationWithJsonResponse(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, JsonElement input, ServiceInfo serviceInfo) throws DoipException {
        try (DoipClientResponse response = performOperation(targetId, operationId, authInfo, attributes, input, serviceInfo)) {
            if (response.getStatus().equals(DoipConstants.STATUS_OK)) {
                try (InDoipMessage in = response.getOutput()) {
                    InDoipSegment firstSegment = InDoipMessageUtil.getFirstSegment(in);
                    if (firstSegment == null) {
                        throw new BadDoipException("Missing input");
                    }
                    if (!firstSegment.isJson()) {
                        throw new DoipException("Missing expected JSON first segment");
                    }
                    JsonElement json = firstSegment.getJson();
                    return json;
                }
            } else {
                throw doipExceptionFromDoipResponse(response);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    /**
     * Performs an operation, looking up the target's service information by handle resolution.
     * No input (beyond attributes) is provided to the operation.
     *
     * @param targetId the object on which to perform the operation
     * @param operationId the operation to perform
     * @param authInfo the authentication to provide
     * @param attributes the attributes to provide to the operation
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes) throws DoipException {
        DoipRequestHeaders headers = headersFrom(targetId, operationId, authInfo, attributes);
        return performOperation(headers, null);
    }

    /**
     * Performs an operation, looking up the target's service information by handle resolution.
     *
     * @param targetId the object on which to perform the operation
     * @param operationId the operation to perform
     * @param authInfo the authentication to provide
     * @param attributes the attributes to provide to the operation
     * @param input the input to the operation as a JsonElement
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, JsonElement input) throws DoipException {
        DoipRequestHeaders headers = headersFrom(targetId, operationId, authInfo, attributes, input);
        return performOperation(headers, null);
    }

    /**
     * Performs an operation, looking up the target's service information by handle resolution.
     *
     * @param targetId the object on which to perform the operation
     * @param operationId the operation to perform
     * @param authInfo the authentication to provide
     * @param attributes the attributes to provide to the operation
     * @param input the input to the operation as an InDoipMessage
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, InDoipMessage input) throws DoipException {
        DoipRequestHeaders headers = headersFrom(targetId, operationId, authInfo, attributes);
        return performOperation(headers, input);
    }

    /**
     * Performs an operation at a specified service.
     * No input (beyond attributes) is provided to the operation.
     *
     * @param targetId the object on which to perform the operation
     * @param operationId the operation to perform
     * @param authInfo the authentication to provide
     * @param attributes the attributes to provide to the operation
     * @param serviceInfo the service at which to perform the operation
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, ServiceInfo serviceInfo) throws DoipException {
        DoipRequestHeaders headers = headersFrom(targetId, operationId, authInfo, attributes);
        return performOperation(headers, null, serviceInfo);
    }

    /**
     * Performs an operation at a specified service.
     *
     * @param targetId the object on which to perform the operation
     * @param operationId the operation to perform
     * @param authInfo the authentication to provide
     * @param attributes the attributes to provide to the operation
     * @param input the input to the operation as a JsonElement
     * @param serviceInfo the service at which to perform the operation
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, JsonElement input, ServiceInfo serviceInfo) throws DoipException {
        DoipRequestHeaders headers = headersFrom(targetId, operationId, authInfo, attributes, input);
        return performOperation(headers, null, serviceInfo);
    }

    /**
     * Performs an operation at a specified service.
     *
     * @param targetId the object on which to perform the operation
     * @param operationId the operation to perform
     * @param authInfo the authentication to provide
     * @param attributes the attributes to provide to the operation
     * @param input the input to the operation as an InDoipMessage
     * @param serviceInfo the service at which to perform the operation
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, InDoipMessage input, ServiceInfo serviceInfo) throws DoipException {
        DoipRequestHeaders headers = headersFrom(targetId, operationId, authInfo, attributes);
        return performOperation(headers, input, serviceInfo);
    }

    /**
     * Performs an operation, looking up the target's service information by handle resolution.
     *
     * @param headers the content of the initial segment of the request
     * @param input the input to the operation as an InDoipMessage
     * @return the response
     */
    @Override
    public DoipClientResponse performOperation(DoipRequestHeaders headers, InDoipMessage input) throws DoipException {
        return performOperation(headers, input, null);
    }

    @Override
    abstract public DoipClientResponse performOperation(DoipRequestHeaders headers, InDoipMessage input, ServiceInfo serviceInfo) throws DoipException;

    /**
     * Creates a digital object at a service.
     *
     * @param dobj the digital object to create
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the created digital object
     */
    @Override
    public DigitalObject create(DigitalObject dobj, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return create(dobj, authInfo, serviceInfo, null);
    }

    /**
     * Creates a digital object at a service.
     *
     * @param dobj the digital object to create
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the created digital object
     */
    @Override
    public DigitalObject create(DigitalObject dobj, AuthenticationInfo authInfo, ServiceInfo serviceInfo, JsonObject attributes) throws DoipException {
        String targetId;
        if (serviceInfo != null && serviceInfo.serviceId != null) {
            targetId = serviceInfo.serviceId;
        } else {
            targetId = "service";
        }
        try (
                InDoipMessage inMessage = buildCreateOrUpdateMessageFrom(dobj, false);
                DoipClientResponse resp = performOperation(targetId, DoipConstants.OP_CREATE, authInfo, attributes, inMessage, serviceInfo);
        ) {
            if (resp.getStatus().equals(DoipConstants.STATUS_OK)) {
                try (InDoipMessage in = resp.getOutput()) {
                    DigitalObject resultDo = digitalObjectFromSegments(in);
                    return resultDo;
                }
            } else {
                throw doipExceptionFromDoipResponse(resp);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    private InDoipMessage buildCreateOrUpdateMessageFrom(DigitalObject dobj, boolean isUpdate) {
        JsonObject dobjJson = GsonUtility.getGson().toJsonTree(dobj).getAsJsonObject();
        List<InDoipSegment> segments = new ArrayList<>();
        InDoipSegment dobjSegment = new InDoipSegmentFromJson(dobjJson);
        segments.add(dobjSegment);
        if (dobj.elements != null) {
            for (Element el : dobj.elements) {
                if (isUpdate && el.in == null) continue;
                JsonObject elementSegmentJson = new JsonObject();
                elementSegmentJson.addProperty("id", el.id);
                InDoipSegment elementHeaderSegment = new InDoipSegmentFromJson(elementSegmentJson);
                segments.add(elementHeaderSegment);
                InDoipSegment elementBytesSegment = new InDoipSegmentFromInputStream(false, el.in);
                segments.add(elementBytesSegment);
            }
        }
        return new InDoipMessageFromCollection(segments);
    }

    /**
     * Updates a digital object.
     *
     * @param dobj the digital object to update
     * @param authInfo the authentication to provide
     * @return the updated digital object
     */
    @Override
    public DigitalObject update(DigitalObject dobj, AuthenticationInfo authInfo) throws DoipException {
        return update(dobj, authInfo, null);
    }

    /**
     * Updates a digital object at a specified service.
     *
     * @param dobj the digital object to update
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the updated digital object
     */
    @Override
    public DigitalObject update(DigitalObject dobj, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return update(dobj, authInfo, serviceInfo, null);
    }

    /**
     * Updates a digital object at a specified service.
     *
     * @param dobj the digital object to update
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the updated digital object
     */
    @Override
    public DigitalObject update(DigitalObject dobj, AuthenticationInfo authInfo, ServiceInfo serviceInfo, JsonObject attributes) throws DoipException {
        try (
                InDoipMessage inMessage = buildCreateOrUpdateMessageFrom(dobj, true);
                DoipClientResponse resp = performOperation(dobj.id, DoipConstants.OP_UPDATE, authInfo, attributes, inMessage, serviceInfo);
        ) {
            if (resp.getStatus().equals(DoipConstants.STATUS_OK)) {
                try (InDoipMessage in = resp.getOutput()) {
                    DigitalObject resultDo = digitalObjectFromSegments(in);
                    return resultDo;
                }
            } else {
                throw doipExceptionFromDoipResponse(resp);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    /**
     * Retrieves a digital object.
     *
     * @param targetId the id of the object to retrieve
     * @param authInfo the authentication to provide
     * @return the digital object
     */
    @Override
    public DigitalObject retrieve(String targetId, AuthenticationInfo authInfo) throws DoipException {
        return retrieve(targetId, false, authInfo, null);
    }

    /**
     * Retrieves a digital object from a specified service.
     *
     * @param targetId the id of the object to retrieve
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the digital object
     */
    @Override
    public DigitalObject retrieve(String targetId, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return retrieve(targetId, false, authInfo, serviceInfo);
    }


    /**
     * Retrieves a digital object, possibly including all element data.
     *
     * @param targetId the id of the object to retrieve
     * @param includeElementData if true, include data for all elements
     * @param authInfo the authentication to provide
     * @return the digital object
     */
    @Override
    public DigitalObject retrieve(String targetId, boolean includeElementData, AuthenticationInfo authInfo) throws DoipException {
        return retrieve(targetId, includeElementData, authInfo, null);
    }

    /**
     * Retrieves a digital object from a specified service, possibly including all element data.
     *
     * @param targetId the id of the object to retrieve
     * @param includeElementData if true, include data for all elements
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the digital object
     */
    @Override
    public DigitalObject retrieve(String targetId, boolean includeElementData, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        JsonObject attributes = new JsonObject();
        if (includeElementData) {
            attributes.addProperty("includeElementData", includeElementData);
        }
        return retrieve(targetId, attributes, authInfo, serviceInfo);
    }

    /**
     * Retrieves a digital object from a specified service, possibly including attributes, like to include all element data or include only features allowed by a filter.
     *
     * @param targetId the id of the object to retrieve
     * @param attributes like to include all element data or include only features allowed by a filter
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the digital object
     */
    @Override
    public DigitalObject retrieve(String targetId, JsonObject attributes, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        try (DoipClientResponse response = performOperation(targetId, DoipConstants.OP_RETRIEVE, authInfo, attributes, serviceInfo)) {
            if (response.getStatus().equals(DoipConstants.STATUS_OK)) {
                try (InDoipMessage in = response.getOutput()) {
                    DigitalObject resultDo = digitalObjectFromSegments(in);
                    return resultDo;
                }
            } else if (response.getStatus().equals(DoipConstants.STATUS_NOT_FOUND)) {
                return null;
            } else {
                throw doipExceptionFromDoipResponse(response);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    /**
     * Deletes a digital object.
     *
     * @param targetId the id of the object to delete
     * @param authInfo the authentication to provide
     */
    @Override
    public void delete(String targetId, AuthenticationInfo authInfo) throws DoipException {
        delete(targetId, authInfo, null);
    }

    /**
     * Deletes a digital object from a specified service.
     *
     * @param targetId the id of the object to delete
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     */
    @Override
    public void delete(String targetId, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        delete(targetId, authInfo, serviceInfo, null);
    }

    /**
     * Deletes a digital object from a specified service.
     *
     * @param targetId the id of the object to delete
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     */
    @Override
    public void delete(String targetId, AuthenticationInfo authInfo, ServiceInfo serviceInfo, JsonObject attributes) throws DoipException {
        JsonElement input = null;
        try (DoipClientResponse resp = performOperation(targetId, DoipConstants.OP_DELETE, authInfo, attributes, input, serviceInfo)) {
            if (resp.getStatus().equals(DoipConstants.STATUS_OK)) {
                return;
            } else {
                throw doipExceptionFromDoipResponse(resp);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    /**
     * Lists operations available for a digital object.
     *
     * @param targetId the id of the digital object
     * @param authInfo the authentication to provide
     * @return the list of available operation ids
     */
    @Override
    public List<String> listOperations(String targetId, AuthenticationInfo authInfo) throws DoipException {
        return listOperations(targetId, authInfo, null);
    }

    /**
     * Lists operations available for a digital object at a specified service.
     *
     * @param targetId the id of the digital object
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the list of available operation ids
     */
    @Override
    public List<String> listOperations(String targetId, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return listOperations(targetId, authInfo, serviceInfo, null);
    }

    /**
     * Lists operations available for a digital object at a specified service.
     *
     * @param targetId the id of the digital object
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the list of available operation ids
     */
    @Override
    public List<String> listOperations(String targetId, AuthenticationInfo authInfo, ServiceInfo serviceInfo, JsonObject attributes) throws DoipException {
        JsonElement input = null;
        try (DoipClientResponse resp = performOperation(targetId, DoipConstants.OP_LIST_OPERATIONS, authInfo, attributes, input, serviceInfo)) {
            if (resp.getStatus().equals(DoipConstants.STATUS_OK)) {
                try (InDoipMessage in = resp.getOutput()) {
                    InDoipSegment firstSegment = InDoipMessageUtil.getFirstSegment(in);
                    if (firstSegment == null) {
                        throw new DoipException("Missing first segment in response");
                    }
                    List<String> results = GsonUtility.getGson().fromJson(firstSegment.getJson(), new TypeToken<List<String>>(){}.getType());
                    return results;
                }
            } else {
                throw doipExceptionFromDoipResponse(resp);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    /**
     * Search for digital objects, returning the ids of the results.
     *
     * @param targetId the id of the operation target (generally a DOIP service id)
     * @param query the query
     * @param params the query parameters
     * @param authInfo the authentication to provide
     * @return the search results as ids
     */
    @Override
    public SearchResults<String> searchIds(String targetId, String query, QueryParams params, AuthenticationInfo authInfo) throws DoipException {
        return searchIds(targetId, query, params, authInfo, null);
    }

    /**
     * Search for digital objects, returning the ids of the results.
     *
     * @param targetId the id of the operation target (generally a DOIP service id)
     * @param query the query
     * @param params the query parameters
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the search results as ids
     */
    @Override
    public SearchResults<String> searchIds(String targetId, String query, QueryParams params, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return searchIdsOrFull("id", String.class, targetId, query, params, authInfo, serviceInfo);
    }

    /**
     * Search for digital objects, returning the full results as digital objects.
     *
     * @param targetId the id of the operation target (generally a DOIP service id)
     * @param query the query
     * @param params the query parameters
     * @param authInfo the authentication to provide
     * @return the search results as digital objects
     */
    @Override
    public SearchResults<DigitalObject> search(String targetId, String query, QueryParams params, AuthenticationInfo authInfo) throws DoipException {
        return search(targetId, query, params, authInfo, null);
    }

    /**
     * Search for digital objects, returning the full results as digital objects.
     *
     * @param targetId the id of the operation target (generally a DOIP service id)
     * @param query the query
     * @param params the query parameters
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the search results as digital objects
     */
    @Override
    public SearchResults<DigitalObject> search(String targetId, String query, QueryParams params, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return searchIdsOrFull("full", DigitalObject.class, targetId, query, params, authInfo, serviceInfo);
    }

    @SuppressWarnings("resource")
    private <T> SearchResults<T> searchIdsOrFull(String type, Class<T> klass, String targetId, String query, QueryParams params, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        DoipClientResponse resp = null;
        try {
            JsonObject attributes = getSearchAttributes(type, query, params);
            resp = performOperation(targetId, DoipConstants.OP_SEARCH, authInfo, attributes, serviceInfo);
            if (resp.getStatus().equals(DoipConstants.STATUS_OK)) {
                return new DoipSearchResults<>(resp, klass);
            } else {
                throw doipExceptionFromDoipResponse(resp);
            }
        } catch (Exception e) {
            closeQuietly(resp);
            if (e instanceof DoipException) throw (DoipException) e;
            throw new DoipException(e);
        }
    }

    private static void closeQuietly(DoipClientResponse response) {
        if (response != null) try { response.close(); } catch (Exception ex) { }
    }

    private static JsonObject getSearchAttributes(String type, String query, QueryParams params) {
        if (params == null) params = QueryParams.DEFAULT;
        JsonObject attributes = new JsonObject();
        attributes.addProperty("query", query);
        attributes.addProperty("pageNum", params.getPageNum());
        attributes.addProperty("pageSize", params.getPageSize());
        if (type == null) {
            type = "full";
        }
        attributes.addProperty("type", type);
        if (params.getSortFields() != null) {
            String sortFields = sortFieldsToString(params.getSortFields());
            if (sortFields != null) {
                attributes.addProperty("sortFields", sortFields);
            }
        }
        try {
            List<FacetSpecification> facets = params.getFacets();
            if (facets != null && !facets.isEmpty()) {
                attributes.add("facets", GsonUtility.getGson().toJsonTree(facets));
            }
        } catch (IncompatibleClassChangeError e) {
            // ignore: facets and filterQueries are not supported by some incompatible version
        }
        return attributes;
    }

    private static String sortFieldsToString(List<SortField> sortFields) {
        if (sortFields != null && !sortFields.isEmpty()) {
            List<String> sortFieldsForTransport = new ArrayList<>(sortFields.size());
            for(SortField sortField : sortFields) {
                if(sortField.isReverse()) sortFieldsForTransport.add(sortField.getName() + " DESC");
                else sortFieldsForTransport.add(sortField.getName());
            }
            if (!sortFieldsForTransport.isEmpty()) {
                return String.join(",", sortFieldsForTransport);
            }
        }
        return null;
    }

    /**
     * Performs the "hello" operation.
     *
     * @param targetId the id of the operation target (generally a DOIP service id)
     * @param authInfo the authentication to provide
     * @return the result of the hello operation as a service info digital object
     */
    @Override
    public DigitalObject hello(String targetId, AuthenticationInfo authInfo) throws DoipException {
        return hello(targetId, authInfo, null);
    }

    /**
     * Performs the "hello" operation at a specified service.
     *
     * @param targetId the id of the operation target (generally a DOIP service id)
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return the result of the hello operation as a service info digital object
     */
    @Override
    public DigitalObject hello(String targetId, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        JsonElement input = null;
        try (DoipClientResponse response = performOperation(targetId, DoipConstants.OP_HELLO, authInfo, null, input, serviceInfo)) {
            if (response.getStatus().equals(DoipConstants.STATUS_OK)) {
                try (InDoipMessage in = response.getOutput()) {
                    DigitalObject resultDo = digitalObjectFromSegments(in);
                    return resultDo;
                }
            } else {
                throw doipExceptionFromDoipResponse(response);
            }
        } catch (DoipException e) {
            throw e;
        } catch (Exception e) {
            throw new DoipException(e);
        }
    }

    /**
     * Retrieves an element from a digital object.
     *
     * @param targetId the id of the digital object
     * @param elementId the id of the element
     * @param authInfo the authentication to provide
     * @return an input stream with the bytes of the element
     */
    @SuppressWarnings("resource")
    @Override
    public InputStream retrieveElement(String targetId, String elementId, AuthenticationInfo authInfo) throws DoipException {
        return retrieveElement(targetId, elementId, authInfo, null);
    }

    /**
     * Retrieves an element from a digital object at a specified service.
     *
     * @param targetId the id of the digital object
     * @param elementId the id of the element
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return an input stream with the bytes of the element
     */
    @SuppressWarnings("resource")
    @Override
    public InputStream retrieveElement(String targetId, String elementId, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return retrieveElement(targetId, elementId, authInfo, serviceInfo, null);
    }

    /**
     * Retrieves an element from a digital object at a specified service.
     *
     * @param targetId the id of the digital object
     * @param elementId the id of the element
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @attributes
     * @return an input stream with the bytes of the element
     */
    @SuppressWarnings("resource")
    @Override
    public InputStream retrieveElement(String targetId, String elementId, AuthenticationInfo authInfo, ServiceInfo serviceInfo, JsonObject attributes) throws DoipException {
        if (attributes == null) {
            attributes = new JsonObject();
        }
        attributes.addProperty("element", elementId);
        DoipClientResponse response = null;
        try {
            response = performOperation(targetId, DoipConstants.OP_RETRIEVE, authInfo, attributes, serviceInfo);
            if (response.getStatus().equals(DoipConstants.STATUS_OK)) {
                InDoipMessage in = response.getOutput();
                InDoipSegment firstSegment = InDoipMessageUtil.getFirstSegment(in);
                if (firstSegment == null) {
                    throw new DoipException("Missing first segment");
                }
                return getElementInputStreamWithCorrectClose(firstSegment, response);
            } else {
                throw doipExceptionFromDoipResponse(response);
            }
        } catch (Exception e) {
            closeQuietly(response);
            if (e instanceof DoipException) throw (DoipException) e;
            throw new DoipException(e);
        }
    }

    @SuppressWarnings("resource")
    @Override
    public InputStream retrievePartialElement(String targetId, String elementId, Long start, Long end, AuthenticationInfo authInfo) throws DoipException {
        return retrievePartialElement(targetId, elementId, start, end, authInfo, null);
    }

    /**
     * Retrieves a byte range of an element from a digital object at a specified service.
     * Either start or end may be null.
     * If neither are null, all bytes from start to end, inclusive, with the first byte of the element being numbered 0, are returned.
     * If both are null, the entire element is returned.
     * If only end is null, all bytes from start to the end of the element are returned.
     * If only start is null, end indicates how many bytes to return from the end of the element. For example, if start is null and end is 500, the
     * last 500 bytes of the element are returned.
     *
     * @param targetId the id of the digital object
     * @param elementId the id of the element
     * @param start the start byte of the desired range, or null (indicates that the number of bytes given by end should be retrieved from the end of the element)
     * @param end the end byte of the desired range, or null (indicates the range should extend to the end of the element)
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return an input stream with the bytes of the element from start to end inclusive
     */
    @SuppressWarnings("resource")
    @Override
    public InputStream retrievePartialElement(String targetId, String elementId, Long start, Long end, AuthenticationInfo authInfo, ServiceInfo serviceInfo) throws DoipException {
        return retrievePartialElement(targetId, elementId, start, end, authInfo, serviceInfo, null);
    }

    /**
     * Retrieves a byte range of an element from a digital object at a specified service.
     * Either start or end may be null.
     * If neither are null, all bytes from start to end, inclusive, with the first byte of the element being numbered 0, are returned.
     * If both are null, the entire element is returned.
     * If only end is null, all bytes from start to the end of the element are returned.
     * If only start is null, end indicates how many bytes to return from the end of the element. For example, if start is null and end is 500, the
     * last 500 bytes of the element are returned.
     *
     * @param targetId the id of the digital object
     * @param elementId the id of the element
     * @param start the start byte of the desired range, or null (indicates that the number of bytes given by end should be retrieved from the end of the element)
     * @param end the end byte of the desired range, or null (indicates the range should extend to the end of the element)
     * @param authInfo the authentication to provide
     * @param serviceInfo the service at which to perform the operation
     * @return an input stream with the bytes of the element from start to end inclusive
     */
    @SuppressWarnings("resource")
    @Override
    public InputStream retrievePartialElement(String targetId, String elementId, Long start, Long end, AuthenticationInfo authInfo, ServiceInfo serviceInfo, JsonObject attributes) throws DoipException {
        if (attributes == null) {
            attributes = new JsonObject();
        }
        attributes.addProperty("element", elementId);
        JsonObject range = new JsonObject();
        range.addProperty("start", start);
        range.addProperty("end", end);
        attributes.add("range", range);
        DoipClientResponse response = null;
        try {
            response = performOperation(targetId, DoipConstants.OP_RETRIEVE, authInfo, attributes, serviceInfo);
            if (response.getStatus().equals(DoipConstants.STATUS_OK)) {
                InDoipMessage in = response.getOutput();
                InDoipSegment firstSegment = InDoipMessageUtil.getFirstSegment(in);
                if (firstSegment == null) {
                    throw new DoipException("Missing first segment");
                }
                return getElementInputStreamWithCorrectClose(firstSegment, response);
            } else {
                throw doipExceptionFromDoipResponse(response);
            }
        } catch (Exception e) {
            closeQuietly(response);
            if (e instanceof DoipException) throw (DoipException) e;
            throw new DoipException(e);
        }
    }

    @SuppressWarnings("resource")
    private static InputStream getElementInputStreamWithCorrectClose(InDoipSegment doipSegment, DoipClientResponse response) {
        InputStream in = doipSegment.getInputStream();
        return new DelegatedCloseableInputStream(in, () -> closeQuietly(response));
    }

    private static DoipRequestHeaders headersFrom(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes, JsonElement input) throws DoipException {
        DoipRequestHeaders headers = new DoipRequestHeaders();
        headers.targetId = targetId;
        headers.operationId = operationId;
        if (authInfo != null) {
            headers.clientId = authInfo.getClientId();
            JsonElement authentication = authInfo.getAuthentication();
            if (authentication != null) {
                headers.authentication = authentication;
            }
        }
        headers.attributes = attributes;
        if (input != null) {
            headers.input = input;
        }
        return headers;
    }

    private static DoipRequestHeaders headersFrom(String targetId, String operationId, AuthenticationInfo authInfo, JsonObject attributes) throws DoipException {
        return headersFrom(targetId, operationId, authInfo, attributes, null);
    }

    DigitalObject digitalObjectFromSegments(InDoipMessage input) throws IOException, DoipException {
        InDoipSegment firstSegment = InDoipMessageUtil.getFirstSegment(input);
        if (firstSegment == null) {
            throw new BadDoipException("Missing input");
        }
        DigitalObject digitalObject = GsonUtility.getGson().fromJson(firstSegment.getJson(), DigitalObject.class);

        if (digitalObject.elements != null) {
            Map<String, Element> elements = new HashMap<>();
            for (Element el : digitalObject.elements) {
                elements.put(el.id, el);
            }
            Iterator<InDoipSegment> segments = input.iterator();
            while (segments.hasNext()) {
                InDoipSegment headerSegment = segments.next();
                String elementId;
                try {
                    elementId = headerSegment.getJson().getAsJsonObject().get("id").getAsString();
                } catch (Exception e) {
                    throw new DoipException("Unexpected element header");
                }
                if (!segments.hasNext()) {
                    throw new DoipException("Unexpected end of input");
                }
                InDoipSegment elementBytesSegment = segments.next();
                Element el = elements.get(elementId);
                if (el == null) {
                    throw new DoipException("No such element " + elementId);
                }
                @SuppressWarnings("resource")
                InputStream in = elementBytesSegment.getInputStream();
                el.in = persistInputStream(in);
            }
        } else {
            if (!InDoipMessageUtil.isEmpty(input)) {
                throw new DoipException("Unexpected input segments");
            }
        }
        return digitalObject;
    }

    private static ByteArrayInputStream persistInputStream(InputStream in) throws IOException {
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        byte[] buf = new byte[8192];
        int r;
        while ((r = in.read(buf)) > 0) {
            bout.write(buf, 0, r);
        }
        return new ByteArrayInputStream(bout.toByteArray());
    }

    public static DoipException doipExceptionFromDoipResponse(DoipClientResponse resp) throws IOException {
        JsonElement response = ErrorMessageUtil.getJsonResponseFromErrorResponse(resp);
        DoipException e = new DoipException(resp.getStatus(), response);
        return e;
    }
}
