package net.dona.doip.client;

import java.util.Collections;
import java.util.List;

/**
 * An object for holding query parameters for {@link DoipClient} search operations.
 */
public class QueryParams {

    /**
     * Default query parameters.  Passing {@code null} to search methods amounts to using this.  No pagination and no sorting.
     */
    public static final QueryParams DEFAULT = new QueryParams();

    private final List<SortField> sortFields;
    private final int pageNum;
    private final int pageSize;
    private final List<FacetSpecification> facets;

    /**
     * Construct default QueryParams, with pageSize = -1 (returns all results).
     * Prefer {@link #DEFAULT}.
     */
    public QueryParams() {
        this(0, -1);
    }

    /**
     * Construct a QueryParams.
     *
     * @param pageNum  the page number to return.  Starts at 0.  Ignored if pageSize &lt;= 0.
     * @param pageSize the number of objects to return.  PageSize of &lt; 0 means return all.
     */
    public QueryParams(int pageNum, int pageSize) {
        this(pageNum, pageSize, Collections.emptyList());
    }

    public QueryParams(int pageNum, int pageSize, List<SortField> sortFields) {
        this(pageNum, pageSize, sortFields, Collections.emptyList());
    }

    /**
     * Construct a QueryParams.
     *
     * @param pageNum  the page number to return.  Starts at 0.  Ignored if pageSize &lt;= 0.
     * @param pageSize the number of objects to return.  PageSize of &lt; 0 means return all.
     * @param sortFields how to sort the results
     * @param facets information about requested faceting
     *
     * When a user specifies a `facets` field in a search request
     * (e.g., `"facets":[{"field":"/bucket"}]` in a JSON object in the body of a POST request),
     * the Cordra indexer provides a `facets` field with an array of facets (e.g., `"facets":[...]`).
     * Each facet contains a `field` field with an array of facets (e.g., `"facets":[...]`).
     * Each facet contains a `field` field with a value `<v_field>` (e.g., `"field":"/bucket"`).
     * facets is a list of objects encapsulating each value <v_field>
     * (e.g., a list containing an object encapsulating "/bucket").
     */

    public QueryParams(int pageNum, int pageSize, List<SortField> sortFields, List<FacetSpecification> facets) {
        this.pageNum = pageNum;
        this.pageSize = pageSize;
        this.sortFields = sortFields;
        this.facets = facets;
    }

    public int getPageNum() {
        return pageNum;
    }

    public int getPageSize() {
        return pageSize;
    }

    public List<SortField> getSortFields() {
        return sortFields;
    }

    public List<FacetSpecification> getFacets() {
        return facets;
    }

    public static Builder builder() {
        return new Builder();
    }

    @SuppressWarnings("hiding")
    public static class Builder {
        private int pageNum = 0;
        private int pageSize = -1;
        private List<SortField> sortFields = Collections.emptyList();
        private List<FacetSpecification> facets = Collections.emptyList();

        public Builder() { }

        public Builder pageNum(int pageNum) {
            this.pageNum = pageNum;
            return this;
        }

        public Builder pageSize(int pageSize) {
            this.pageSize = pageSize;
            return this;
        }

        public Builder sortFields(List<SortField> sortFields) {
            this.sortFields = sortFields;
            return this;
        }

        public Builder facets(List<FacetSpecification> facets) {
            this.facets = facets;
            return this;
        }

        public QueryParams build() {
            return new QueryParams(pageNum, pageSize, sortFields, facets);
        }
    }
}
